<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportObjects\ValueMapper;

use ExportObjects\Api\Manager as ApiManager;
use ExportObjects\Config\Client;
use ExportObjects\Config\Objects\Property as ConfigProperty;
use ExportObjects\Mapper\Mapper as ObjectMapper;

/**
 * Class Temperature (°C, °F)
 *
 * @package ExportObjects\ValueMapper
 * @author Oliver Friedrich <of@life-style.de>
 */
class Temperature extends Base
{

    /**
     * @var string
     */
    private $baseUnit;

    /**
     * Temperature unit mapper constructor.
     *
     * @param ApiManager $apiM
     * @param Client $client
     */
    public function __construct(ApiManager $apiM, Client $client)
    {
        parent::__construct($apiM, $client);
        $this->validUnits = array(
            '°C' => 'C',
            'C' => 'C',
            '°F' => 'F',
            'F' => 'F',
        );
        $this->baseUnit = $this->getUnit($client->getUnits()->getTemperature()->getBase());
    }

    /**
     * Map value
     *
     * Y = 9/5X +32 (Y= Fahrenheit; X = Celsius)
     * X = 5/9(Y - 32) (Y = Fahrenheit; X = Celsius)
     *
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     * @param string         $language
     *
     * @return mixed
     * @throws InvalidUnitException
     */
    public function map(ObjectMapper $objectMapper, ConfigProperty $property, $language)
    {
        $value = trim($this->getObjectValue($objectMapper, $property));

        // Do not map empty values
        if (!strlen($value)) {
            return '';
        }

        $cleanValue = $this->cleanUpValue($value);
        $cleanUnit = $this->getUnit($this->getObjectUnit($objectMapper, $property));

        if ($this->baseUnit == $cleanUnit) {
            return $cleanValue;
        }

        // Convert Celsius into Fahrenheit
        if ('F' == $this->baseUnit) {
            return 9 / 5 * $cleanValue + 32;
        }

        // Convert Fahrenheit into Celsius
        return 5 / 9 * ($cleanValue - 32);
    }

}
