<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportObjects\Worker;

use ExportObjects\Api\Manager as ApiManager;
use ExportObjects\Config\Client;
use ExportObjects\Mapper\Object as ObjectMapper;
use ExportObjects\ValueMapper\MapperInterface as ValueMapper;
use Pimcore\Model\Object\Classificationstore;
use Pimcore\Model\Object\Concrete;
use Pimcore\Model\Object\Fieldcollection;
use Swoox\Api\Model\Request\Index\Swoox\Item;
use Swoox\Api\Model\Request\Index\Swoox\Request;

/**
 * Update-Worker
 *
 * @package ExportObjects\Worker
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Update
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {

        $this->apiM = $apiM;
    }

    /**
     * Run export
     *
     * @param Concrete $object
     */
    public function run(Concrete $object)
    {

        /**
         * Object not published
         */
        if (!$object->isPublished()) {
            $this->apiM->logger()->debug(sprintf('Object "%s" (ID %d) not exported: not published', $object->getKey(), $object->getId()));

            return;
        }

        /**
         * Initialize object mapper
         */
        $mapper = $this->apiM->mapper()->object($object);

        foreach ($this->apiM->clients()->active() as $client) {

            /**
             * Prepare request
             */
            $modelRequest = $this->apiM->model()->request()->index()->swoox();

            /**
             * Prepare data
             */
            $this->update($client, $modelRequest, $mapper);

            $this->apiM->logger()->debug(sprintf('Object "%s" (ID %d) added to update request', $object->getKey(), $object->getId()));

            /**
             * Send request
             */
            $service = $this->apiM->clients()->service($client, 'Objects::Update');
            $this->apiM->webservice()->request()->send($modelRequest, $client, $service);
        }
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param              $propertyName
     * @param              $language
     */
    public function addLocalImagePath(ObjectMapper $objectMapper, Item $item, $propertyName, $language)
    {

        /**
         * @var $imageObject \Pimcore\Model\Asset\Image
         */
        $imageObject = $objectMapper->getValue($propertyName);

        $imageUrl = $imageObject->getFullPath();

        /**
         * Add node
         */
        $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList();
        $nameFieldList
            ->setName($propertyName)
            ->setValue($imageUrl);

        $documentFields = $item->getDocumentFields();
        $documentFields[] = $nameFieldList;
        $item->setDocumentFields($documentFields);
    }

    /**
     * Update datasheet
     *
     * Prepare each attribute in each language even if it is an attribute
     * which is language independent. This makes live easier on the other site.
     *
     * @param Client       $client
     * @param Request      $request
     * @param ObjectMapper $objectMapper
     */
    private function update(Client $client, Request $request, ObjectMapper $objectMapper)
    {

        $item = $this->apiM->model()->request()->index()->item();

        $item->setDocumentUniqueKey($objectMapper->getObject()->getId());

        /**
         * gets overwritten by config
         */
        $item->setDocumentLanguage('de_DE');
        $item->setDocumentPrefix('');
        $item->setDocumentRanking(1);
        $item->setDocumentText('');
        $item->setDocumentTitle('');
        $item->setDocumentURL('');

        $language = 'de';
        foreach ($objectMapper->getObjectProperties() as $propertyName) {

            /**
             * check and calculate types
             */
            if ('localizedfields' == $propertyName) {

                /**
                 * Localized fields
                 */
                foreach ($objectMapper->getLocalizedValues($language) as $key => $value) {
                    $this->addLocalizedValue($objectMapper, $item, $key, $value, $language);
                }

            } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Object\Fieldcollection')) {

                /**
                 * Field collection
                 */
                $this->addFieldcollectionValues($objectMapper, $item, $objectMapper->getValue($propertyName), $language);

            } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Asset\Image')) {

                /**
                 * pimcore image path
                 */
                $this->addLocalImagePath($objectMapper, $item, $propertyName, $language);

            } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Asset')) {

                /**
                 * skip assets
                 */

            } elseif (is_array($objectMapper->getValue($propertyName))) {

                /**
                 * array
                 */
                $this->addArrayValues($objectMapper, $item, $propertyName, $language);

            } elseif (is_scalar($objectMapper->getValue($propertyName))) {

                /**
                 * Plain value
                 */
                $this->addValue($objectMapper, $item, $propertyName, $language);

            } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Object\Classificationstore')) {

                /**
                 * Classificationstore
                 */
                $this->addClassificationStore($objectMapper->getValue($propertyName), $item);

            } elseif ($objectMapper->getValue($propertyName) === null) {

                /**
                 * skip null
                 */

            } else {

                /**
                 * log non processed types
                 */
                dump($objectMapper->getValue($propertyName));

            }
        }

        $this->addParentAttributes($objectMapper, $item);

        /**
         * Cycle through configured languages
         */
//        foreach ($this->apiM->pimcore()->config()->getLanguages() as $language) {
//
//            // Add language
//            //$item->addDocumentLanguage($language);
//
//            // Switch language
//            //$this->apiM->translate()->translator()->setLocale($language);
//
//            // Add properties
//            foreach ($objectMapper->getObjectProperties() as $propertyName) {
//
//                if ('localizedfields' == $propertyName) {
//
//                    // Localized fields
//                    foreach ($objectMapper->getLocalizedValues($language) as $key => $value) {
//                        $this->addLocalizedValue($objectMapper, $item, $key, $value, $language);
//                    }
//
//                } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Object\Fieldcollection')) {
//
//                    // Field collection
//                    $this->addFieldcollectionValues($objectMapper, $item, $objectMapper->getValue($propertyName), $language);
//
//                } else {
//
//                    // Plain value
//                    $this->addValue($client, $objectMapper, $item, $propertyName, $language);
//
//                }
//            }
//        }

        $request->addItem($item);
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param string       $key
     * @param string       $value
     * @param string       $language
     */
    private function addLocalizedValue(ObjectMapper $objectMapper, Item $item, $key, $value, $language)
    {

        $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList()
            ->setName($key . '_' . $language)
            #->setLanguage($language)
            #->setLabelId($objectMapper->mapNameId($key))
            #->setLabel($this->apiM->translate()->translator()->trans($objectMapper->mapNameId($key)))
            ->setValue($value);

        $item->addDocumentFields($nameFieldList);
    }

    /**
     * @param ObjectMapper    $objectMapper
     * @param Item            $item
     * @param Fieldcollection $fieldCollection
     * @param string          $language
     */
    private function addFieldcollectionValues(ObjectMapper $objectMapper, Item $item, Fieldcollection $fieldCollection, $language)
    {

        #$translator = $this->apiM->translate()->translator();

        /**
         * Add field collection
         */
        foreach ($fieldCollection->getItems() as $fieldCollectionData) {

            /**
             * Field collection mapper
             */
            $fieldCollectionMapper = $this->apiM->mapper()->fieldCollection($fieldCollectionData);

            /**
             * Field collection data - item
             */
            foreach ($fieldCollectionMapper->getValues() as $key => $values) {
                $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList();

                /**
                 * Each item in field collection data may have multiple values selected
                 */
                if (is_array($values)) {
                    foreach ($values as $value) {
                        $valueItem = $this->apiM->model()->request()->index()->valueItem();

                        $valueItem->setValue($value);

                        $nameFieldList
                            ->setName($key)
                            ->addValues($valueItem);
                    }

                } else {
                    $nameFieldList
                        ->setName($key)
                        ->setValue($values);
                }

                $item->addDocumentFields($nameFieldList);
            }
        }
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param string       $propertyName
     * @param string       $language
     */
    private function addArrayValues(ObjectMapper $objectMapper, Item $item, $propertyName, $language)
    {

        #$translator = $this->apiM->translate()->translator();

        $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList();
        $nameFieldList->setName($propertyName);

        /**
         * Add field collection
         */
        foreach ($objectMapper->getValue($propertyName) as $value) {
            $valueItem = $this->apiM->model()->request()->index()->valueItem();
            $valueItem->setValue($value);

            $nameFieldList->addValues($valueItem);
        }

        $item->addDocumentFields($nameFieldList);
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param string       $propertyName
     * @param string       $language
     */
    private function addValue(ObjectMapper $objectMapper, Item $item, $propertyName, $language)
    {

        /**
         * @todo fix or remove
         */
        #$translator = $this->apiM->translate()->translator();
        $propertyMapped = false;

        $this->setDocumentValues($item, $propertyName, trim($objectMapper->getValue($propertyName)));

        foreach ($this->apiM->getExportConfig()->getProperties() as $property) {

            /**
             * Property does not match
             */
            if ($property->getId() != $propertyName) {
                continue;
            }

            /**
             * Ignore property
             */
            if ($property->isDisabled()) {
                return;
            }

            /**
             * Map value
             *
             * @todo implement mapper
             */
            $value = trim($objectMapper->getValue($propertyName));
            #$value = trim($this->valueMapper($client, $property->getMapper())->map($objectMapper, $property, $language));

            /**
             * Skip empty value
             */
            if (!strlen($value)) {
                continue;
            }

            /**
             * Add node
             */
            $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList();
            $nameFieldList
                ->setName($propertyName)
                ->setValue($value);

            $documentFields = $item->getDocumentFields();
            $documentFields[] = $nameFieldList;
            $item->setDocumentFields($documentFields);

            /**
             * Mark property as mapped
             */
            $propertyMapped = true;
        }

        /**
         * Passthrough not mapped value
         */
        if (!$propertyMapped && strlen($value = trim($objectMapper->getValue($propertyName)))) {
            $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList();
            $nameFieldList
                ->setName($propertyName)
                ->setValue($value);

            $item->addDocumentFields($nameFieldList);
        }
    }

    /**
     * @param Client $client
     * @param        $mapper
     *
     * @return ValueMapper
     */
    private function valueMapper(Client $client, $mapper)
    {

        return $this->apiM->valuemapper()->$mapper($client);
    }

    /**
     * @param Item   $item
     * @param string $propertyName
     * @param string $value
     */
    private function setDocumentValues(Item $item, $propertyName, $value)
    {

        $exportDocumentConfig = $this->apiM->getExportConfig()->getDocument();

        $language = $exportDocumentConfig->getLanguage();
        if ($propertyName == $language && !empty($language)) {
            $item->setDocumentLanguage($value);
        }

        $prefix = $exportDocumentConfig->getPrefix();
        if ($propertyName == $prefix && !empty($prefix)) {
            $item->setDocumentPrefix($value);
        }

        $ranking = $exportDocumentConfig->getRanking();
        if ($propertyName == $ranking && !empty($ranking)) {
            $item->setDocumentRanking($value);
        }

        $title = $exportDocumentConfig->getTitle();
        if ($propertyName == $title && !empty($title)) {
            $item->setDocumentTitle($value);
        }

        $url = $exportDocumentConfig->getUrl();
        if ($propertyName == $url && !empty($url)) {
            $item->setDocumentURL($value);
        }

        $text = $exportDocumentConfig->getText();
        if ($propertyName == $text && !empty($text)) {
            $item->setDocumentText($value);
        }
    }

    /**
     * addClassificationStore
     *
     * @param Classificationstore $classificationStore
     * @param Item                $item
     */
    private function addClassificationStore(Classificationstore $classificationStore, Item $item)
    {

        foreach ($classificationStore->getItems() as $groupId => $group) {
            foreach ($group as $keyId => $key) {
                $keyName = Classificationstore\KeyConfig::getById($keyId)->getTitle();

                $value = $classificationStore->getLocalizedKeyValue($groupId, $keyId, 'de');

                $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList();
                $nameFieldList
                    ->setName($keyName)
                    ->setMeta('attribute');

                if (is_array($value)) {
//                    foreach ($value as $val) {
//                        $valueItem = $this->apiM->model()->request()->index()->valueItem();
//                        $valueItem->setValue($val);
//
//                        $nameFieldList->addValues($valueItem);
//                    }
                } else {
                    $nameFieldList
                        ->setValue($value);
                }

                $documentFields = $item->getDocumentFields();
                $documentFields[] = $nameFieldList;
                $item->setDocumentFields($documentFields);
            }
        }
    }

    /**
     * getParentObject
     *
     * @param \Pimcore\Model\Object\AbstractObject $object
     * @param string                               $type
     *
     * @return \Pimcore\Model\Object\AbstractObject
     */
    private function getParentObject($object, $type)
    {

        $parent = $object->getParent();

        if ($parent === null || $type == '') {
            return null;
        }

        if (is_a($parent, $type)) {
            return $parent;
        }

        return $this->getParentObject($parent, $type);
    }

    /**
     * addParentAttributes
     *
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     */
    private function addParentAttributes(ObjectMapper $objectMapper, Item $item)
    {

        if ($this->apiM->getExportConfig()->getParentObject() !== null) {
            /**
             * @var \Pimcore\Model\Object\Concrete $parentObject
             */
            $parentObject = $this->getParentObject($objectMapper->getObject(), $this->apiM->getExportConfig()->getParentObject()->getClass());

            $attributes = $this->apiM->getExportConfig()->getParentObject()->getAttribute();

            foreach ($attributes as $attribute) {
                $value = $parentObject->getValueForFieldName($attribute->getName());

                $metaData = null;
                $metaDomain = '';
                $domain = '';
                if ($attribute->getMetaData()) {
                    $metaData = $parentObject->getValueForFieldName($attribute->getMetaData());

                    if (is_a($parentObject->getClass()->getFieldDefinition($attribute->getMetaData()), 'Pimcore\Model\Object\ClassDefinition\Data\Image') && $attribute->getDomain()) {
                        $metaDomain = $attribute->getDomain();
                    }
                }

                if (is_a($parentObject->getClass()->getFieldDefinition($attribute->getName()), 'Pimcore\Model\Object\ClassDefinition\Data\Image') && $attribute->getDomain()) {
                    $domain = $attribute->getDomain();
                }

                if (!empty($value)) {
                    $nameFieldList = $this->apiM->model()->request()->index()->nameFieldList();
                    $nameFieldList
                        ->setName($attribute->getName())
                        ->setValue($domain . $value);

                    if (!empty($metaData)) {
                        $nameFieldList->setMeta($metaDomain . $metaData);
                    }

                    $item->addDocumentFields($nameFieldList);
                }
            }
        }
    }
}
