<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Publikat\Pimcore\ExportProductGroupBundle\Config\Objects;

use Doctrine\Common\Collections\ArrayCollection;
use JMS\Serializer\Annotation as Serializer;

/**
 * Class Property
 *
 * @package Publikat\Pimcore\ExportProductGroupBundle\Config\Objects
 */
class Property
{
    /**
     * Id - property name in pimcore object
     *
     * @Serializer\SerializedName("Id")
     * @Serializer\Type("string")
     * @var string
     */
    private $id;

    /**
     * Label - label name for translation
     *
     * @Serializer\SerializedName("Label")
     * @Serializer\Type("string")
     * @var string
     */
    private $label;

    /**
     * Label - label-id in export xml [optional]
     *
     * @Serializer\SerializedName("LabelId")
     * @Serializer\Type("string")
     * @var string
     */
    private $labelId;

    /**
     * Format - output format for sprintf [optional]
     *
     * @Serializer\SerializedName("Format")
     * @Serializer\Type("string")
     * @var string
     */
    private $format = '%s';

    /**
     * Mapper - value mapper [optional]
     *
     * @Serializer\SerializedName("Mapper")
     * @Serializer\Type("string")
     * @var string
     */
    private $mapper = 'format';

    /**
     * Unit - unit property name in datenblatt-object for mapper [optional]
     *
     * @Serializer\SerializedName("Unit")
     * @Serializer\Type("string")
     * @var string
     */
    private $unit;

    /**
     * Unit - unit value for mapper [optional]
     *
     * Use this as alternative to $unit, if no unit-option-select is configured
     *
     * @Serializer\SerializedName("UnitValue")
     * @Serializer\Type("string")
     * @var string
     */
    private $unitValue;

    /**
     * BaseUnit - base unit for mapper [optional]
     *
     * @Serializer\SerializedName("BaseUnit")
     * @Serializer\Type("string")
     * @var string
     */
    private $baseUnit;

    /**
     * Values - list of value property names in datenblatt-object passed to sprintf
     *
     * @Serializer\SerializedName("Value")
     * @Serializer\Type("ArrayCollection<string>")
     * @Serializer\XmlList(inline=true, entry="Value")
     * @var ArrayCollection
     */
    private $values;

    /**
     * Skip - set to true, to ignore property
     *
     * @Serializer\SerializedName("Disabled")
     * @Serializer\Type("boolean")
     * @var bool
     */
    private $disabled = false;

    /**
     * @return string
     */
    public function getBaseUnit()
    {
        return $this->baseUnit;
    }

    /**
     * @return string
     */
    public function getFormat()
    {
        return $this->format;
    }

    /**
     * @return string
     */
    public function getLabelId()
    {
        return $this->labelId ? $this->labelId : $this->getLabel();
    }

    /**
     * @return string
     */
    public function getLabel()
    {
        return $this->label ? $this->label : lcfirst(preg_replace('/^datenblatt/', '', $this->getId()));
    }

    /**
     * @return string
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getMapper()
    {
        return $this->mapper;
    }

    /**
     * @return string
     */
    public function getUnit()
    {
        return $this->unit;
    }

    /**
     * @return string
     */
    public function getUnitValue()
    {
        return $this->unitValue;
    }

    /**
     * @return ArrayCollection
     */
    public function getValues()
    {
        return $this->values;
    }

    /**
     * @return boolean
     */
    public function isDisabled()
    {
        return $this->disabled;
    }

    /**
     * @Serializer\PostDeserialize
     */
    public function updateValues()
    {
        if (null === $this->values || $this->values->isEmpty()) {
            $this->values = new ArrayCollection(array($this->getId()));
        }
    }
}
