<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Publikat\Pimcore\ExportProductGroupBundle\ValueMapper;

use Publikat\Pimcore\ExportProductGroupBundle\Api\Manager as ApiManager;
use Publikat\Pimcore\ExportProductGroupBundle\Config\Client;
use Publikat\Pimcore\ExportProductGroupBundle\Config\Objects\Property as ConfigProperty;
use Publikat\Pimcore\ExportProductGroupBundle\Mapper\ProduktGruppe as ObjectMapper;

/**
 * Unit mapper base class
 *
 * @package Publikat\Pimcore\ExportProductGroupBundle\ValueMapper
 * @author  Oliver Friedrich <of@life-style.de>
 */
abstract class Base implements MapperInterface
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var array
     */
    protected $validUnits = array();

    /**
     * Power constructor.
     *
     * @param ApiManager $apiM
     * @param Client     $client
     */
    public function __construct(ApiManager $apiM, Client $client)
    {
        $this->apiM = $apiM;
        $this->client = $client;
    }

    /**
     * @param string $value
     *
     * @return float|integer
     */
    protected function cleanUpValue($value)
    {
        return trim(str_replace(',', '.', preg_replace('/[^\d^,^\.]/', '', $value)), '.') + 0;
    }

    /**
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     *
     * @return mixed|string
     * @throws InvalidPropertyException
     * @throws InvalidValueException
     */
    protected function getObjectUnit(ObjectMapper $objectMapper, ConfigProperty $property)
    {
        return $property->getUnit() ? $this->getObjectProperty($objectMapper, $property->getUnit()) : $property->getUnitValue();
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param              $propertyName
     *
     * @return mixed
     * @throws InvalidPropertyException
     * @throws InvalidValueException
     */
    private function getObjectProperty(ObjectMapper $objectMapper, $propertyName)
    {
        // Property should not be empty
        if (!strlen($propertyName)) {
            throw new InvalidPropertyException('Cannot get object property! Property is empty. Please check your config.xml of ExportProductGroup-plugin for missing unit/unitValue-tags.');
        }

        $value = $objectMapper->getValue($propertyName);

        // Value should not be an object
        if (is_object($value)) {
            throw new InvalidValueException('Cannot map object property! Value should be a scalar but is object for property ' . $propertyName);
        }

        return $value;
    }

    /**
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     *
     * @return mixed|string
     * @throws InvalidPropertyException
     * @throws InvalidValueException
     */
    protected function getObjectValue(ObjectMapper $objectMapper, ConfigProperty $property)
    {
        return $property->getValues()->isEmpty() ? '' : $this->getObjectProperty($objectMapper, $property->getValues()->first());
    }

    /**
     * @param string $unit
     *
     * @return string
     * @throws InvalidUnitException
     */
    protected function getUnit($unit)
    {
        if (!isset($this->validUnits[$unit])) {
            throw new InvalidUnitException();
        }

        return $this->validUnits[$unit];
    }

    /**
     * Get list of valid units
     *
     * @return array
     */
    public function getValidUnits()
    {
        return array_keys($this->validUnits);
    }
}
