<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Publikat\Pimcore\ExportProductGroupBundle\Worker;

use Pimcore\Event\Model\DataObjectEvent;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\Element\ElementInterface;
use Pimcore\Model\DataObject\Fieldcollection;
use Pimcore\Model\DataObject\ProduktGruppe;
use Publikat\Pimcore\ExportProductGroupBundle\Api\Manager as ApiManager;
use Publikat\Pimcore\ExportProductGroupBundle\Config\Client;
use Publikat\Pimcore\ExportProductGroupBundle\Mapper\ProduktGruppe as ObjectMapper;
use Publikat\Pimcore\ExportProductGroupBundle\Model\Request\Objects\Update\Item;
use Publikat\Pimcore\ExportProductGroupBundle\Model\Request\Objects\Update\Request;
use Publikat\Pimcore\ExportProductGroupBundle\ValueMapper\MapperInterface as ValueMapper;

/**
 * Update-Worker
 *
 * @package Publikat\Pimcore\ExportProductGroupBundle\Worker
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Update
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
    }

    /**
     * @param DataObjectEvent $event
     *
     * @throws \Publikat\Pimcore\ExportProductGroupBundle\Config\InvalidConfigException
     * @throws \ReflectionException
     * @throws \Exception
     */
    public function run(DataObjectEvent $event)
    {
        /** @var ProduktGruppe|ElementInterface|Concrete $object */
        $object = $event->getElement();

        // Object not published
        if (!$object->isPublished()) {
            $this->apiM->logger()->debug(sprintf('Object "%s" (ID %d) not exported: not published', $object->getKey(), $object->getId()));
            return;
        }

        // STYLEFILE-5486 : Dont' run export if only save new version is triggered within Pimcore
        if ($event->hasArgument('saveVersionOnly') && $event->getArgument('saveVersionOnly')) {
            return;
        }

        // Initialize object mapper
        $mapper = $this->apiM->mapper()->produktGruppe($object);

        foreach ($this->apiM->clients()->active() as $client) {
            // Prepare request
            $modelRequest = $this->apiM->model()->objectsUpdate()->request();

            // Prepare data
            $this->update($client, $modelRequest, $mapper);

            $this->apiM->logger()->debug(
                sprintf(
                    'Object "%s" (ID %d) added to update request',
                    $object->getKey(),
                    $object->getId()
                )
            );

            // Send request
            $service = $this->apiM->clients()->service($client, 'Objects::Update');
            $this->apiM->webservice()->request()->send($modelRequest, $client, $service);

            /** mule data */
            $data = $this->apiM->webservice()->request()->getPayload($modelRequest, $service);
            if (null !== $this->apiM->muleDispatcher() && in_array($object->getDisplayShopGruppe(), ['STYLEF', 'MIXED'])) {
                $this->apiM->muleDispatcher()->dispatch($data, 'ProductGroup');
            } else {
                $this->apiM->logger()->warning('Mule Message Buffer service is not available, Data is not put on the Buffer.');
            }
        }
    }

    /**
     * Update datasheet
     *
     * Prepare each attribute in each language even if it is an attribute
     * which is language independent. This makes live easier on the other site.
     *
     * @param Client       $client
     * @param Request      $request
     * @param ObjectMapper $objectMapper
     *
     * @throws \ReflectionException
     * @throws \Exception
     */
    private function update(Client $client, Request $request, ObjectMapper $objectMapper)
    {
        $item = $this->apiM->model()->objectsUpdate()->item($request);
        $item->setProductId($objectMapper->getObject()->getKey());

        // Cycle through configured languages
        foreach ($this->apiM->pimcore()->config()->getLanguages() as $language) {
            // Add language
            $item->addDocumentLanguage($language);

            // Switch language
            $this->apiM->translate()->translator()->setLocale($language);

            // Add properties
            foreach ($objectMapper->getObjectProperties() as $propertyName) {
                if ('localizedfields' == $propertyName) {
                    // Localized fields
                    foreach ($objectMapper->getLocalizedValues($language) as $key => $value) {
                        if (!empty($value)) {
                            $this->addLocalizedValue($objectMapper, $item, $key, $value, $language);
                        }
                    }
                } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Object\Fieldcollection')) {
                    // Field collection
                    $this->addFieldcollectionValues($objectMapper, $item, $objectMapper->getValue($propertyName), $language);
                } else {
                    // Plain value
                    $this->addValue($client, $objectMapper, $item, $propertyName, $language);
                }
            }
        }
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param string       $key
     * @param string       $value
     * @param string       $language
     *
     * @throws \Exception
     */
    private function addLocalizedValue(ObjectMapper $objectMapper, Item $item, $key, $value, $language)
    {
        $shopGroup = $this->determineShopGroup($key);

        $nameFieldList = $this->apiM->model()->objectsUpdate()->nameFieldList($item)
            ->setName($key)
            ->setLanguage($language)
            ->setLabelId($objectMapper->mapNameId($key))
            ->setLabel($this->apiM->translate()->translator()->trans($objectMapper->mapNameId($key)))
            ->setValue($value);

        if ($shopGroup != '') {
            $nameFieldList->setShopGroup($shopGroup);
        }
    }

    /**
     * assign and flag shop specific properties (stylefile, bigtree)
     *
     * @param string $fieldName
     *
     * @return string
     */
    private function determineShopGroup($fieldName)
    {
        if (strpos($fieldName, 'Stylefile') !== false) {
            return 'STYLEF';
        }

        if (strpos($fieldName, 'Bigtree') !== false) {
            return 'BIGT';
        }

        return '';
    }

    /**
     * @param ObjectMapper    $objectMapper
     * @param Item            $item
     * @param Fieldcollection $fieldCollection
     * @param string          $language
     *
     * @throws \ReflectionException
     * @throws \Exception
     */
    private function addFieldcollectionValues(ObjectMapper $objectMapper, Item $item, Fieldcollection $fieldCollection, $language)
    {
        $translator = $this->apiM->translate()->translator();
        $shopGroup = $this->determineShopGroup($fieldCollection->getFieldname());

        // Add field collection
        foreach ($fieldCollection->getItems() as $fieldCollectionData) {
            // Field collection mapper
            $fieldCollectionMapper = $this->apiM->mapper()->fieldCollection($fieldCollectionData);

            // Field collection data - item
            foreach ($fieldCollectionMapper->getValues() as $key => $values) {
                $nameFieldList = $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                    ->setName($key)
                    ->setLanguage($language)
                    ->setLabelId($objectMapper->mapNameId($key))
                    ->setLabel($translator->trans($objectMapper->mapNameId($key)));

                if ($shopGroup != '') {
                    $nameFieldList->setShopGroup($shopGroup);
                }

                // Each item in field collection data may have multiple values selected
                foreach ($values as $value) {
                    $this->apiM->model()->objectsUpdate()->valueItem($nameFieldList)
                        ->setValueId($value)
                        ->setValue($translator->trans($value));
                }
            }
        }
    }

    /**
     * @param Client       $client
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param string       $propertyName
     * @param string       $language
     *
     * @throws \Exception
     */
    private function addValue(Client $client, ObjectMapper $objectMapper, Item $item, $propertyName, $language)
    {
        $translator = $this->apiM->translate()->translator();
        $propertyMapped = false;

        foreach ($this->apiM->pluginConfig()->getObjects()->getProduktGruppe()->getProperties() as $property) {
            // Property does not match
            if ($property->getId() != $propertyName) {
                continue;
            }

            // Ignore property
            if ($property->isDisabled()) {
                return;
            }

            // Map value
            $value = trim($this->valuemapper($client, $property->getMapper())->map($objectMapper, $property, $language));

            // Skip empty value
            if (!strlen($value)) {
                continue;
            }

            // Add node
            $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($property->getLabelId())
                ->setLabel($translator->trans($property->getLabel()))
                ->setValue($translator->trans($value));

            // Mark property as mapped
            $propertyMapped = true;
        }

        // Passthrough not mapped value
        if (!$propertyMapped && strlen($value = trim($objectMapper->getValue($propertyName)))) {
            $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($propertyName)
                ->setLabel($translator->trans($objectMapper->mapNameId($propertyName)))
                ->setValue($translator->trans($value));
        }
    }

    /**
     * @param Client $client
     * @param        $mapper
     *
     * @return ValueMapper
     */
    private function valueMapper(Client $client, $mapper)
    {
        return $this->apiM->valuemapper()->$mapper($client);
    }
}
