<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace ExportProductGroup\Translate;

use ExportProductGroup\Api\Manager as ApiManager;
use Pimcore\Model\Object\AttributUebersetzung;
use Pimcore\Model\Object\AttributUebersetzung\Listing as AttributUebersetzungListing;
use Symfony\Component\Translation\Loader\ArrayLoader as CoreArrayLoader;
use Symfony\Component\Translation\MessageSelector as CoreMessageSelector;
use Symfony\Component\Translation\Translator as CoreTranslator;

/**
 * Translator
 *
 * @package ExportProductGroup\Translate
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Translator
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var array
     */
    private $translator;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {

        $this->apiM = $apiM;
    }

    /**
     * @return string
     */
    public function getLocale()
    {

        return $this->translator()->getLocale();
    }

    /**
     * @return CoreTranslator
     */
    public function translator()
    {

        return null !== $this->translator ? $this->translator : ($this->translator = $this->createTranslator());
    }

    /**
     * @return CoreTranslator
     */
    private function createTranslator()
    {

        $defaultLanguage = $this->apiM->pimcore()->config()->getDefaultLanguage();
        $translator = new CoreTranslator($defaultLanguage, new CoreMessageSelector());
        $translator->addLoader('array', new CoreArrayLoader());

        $translations = array();

        $attributes = new AttributUebersetzungListing();
        foreach ($attributes as $attribute) {
            foreach ($this->apiM->pimcore()->config()->getLanguages() as $language) {
                /* @var AttributUebersetzung $attribute */
                $attribute->isPublished() ? $translations[$language][strtolower($attribute->getKey())] = $attribute->getAttributUebersetzungText($language) : null;
            }
        }

        foreach ($translations as $language => $translation) {
            $translator->addResource('array', $translation, $language);
        }

        $translator->setFallbackLocales($this->apiM->pimcore()->config()->getFallbackLanguages($defaultLanguage));

        return $translator;
    }

    /**
     * @param string $locale
     *
     * @return $this
     */
    public function setLocale($locale)
    {

        $this->translator()->setLocale($locale);
        $this->translator()->setFallbackLocales($this->apiM->pimcore()->config()->getFallbackLanguages($locale));

        return $this;
    }

    /**
     * Translate message
     *
     * @param string $message
     *
     * @return string
     */
    public function trans($message)
    {

        $translated = $this->translator()->trans(strtolower($message));

        return $translated === strtolower($message) ? $message : $translated;
    }
}
