<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace ExportProductGroup\Webservice;

use ExportProductGroup\Api\Manager as ApiManager;
use ExportProductGroup\Config\Client as ClientConfig;
use ExportProductGroup\Config\Service as ServiceConfig;
use ExportProductGroup\Model\Request\RequestInterface;
use ExportProductGroup\Model\Response\ResponseInterface;
use Guzzle\Http\Exception\RequestException as WebserviceRequestException;

/**
 * Webservice-Request
 *
 * @package ExportProductGroup\Webservice
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Request
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {

        $this->apiM = $apiM;
    }

    /**
     * @param RequestInterface $request
     * @param ClientConfig     $client
     * @param ServiceConfig    $service
     *
     * @return ResponseInterface
     * @throws \Exception
     */
    public function send(RequestInterface $request, ClientConfig $client, ServiceConfig $service)
    {

        // Prepare request
        $options = array(
            'headers' => array(
                'Accept' => $service->getAcceptContentType(),
                'Content-Type' => $service->getContentType()
            ),
            'body' => $this->apiM->serializer()->serialize($request->getRequest(), $service->getFormat())
        );

        $this->apiM->logger()->info(sprintf('Object export: sending request to client "%s" url %s', $client->getId(), $service->getUrl()));
        $this->apiM->logger()->debug($options['body']);

        // Send request
        try {
            $webserviceResponse = $this->apiM->webservice()->client()->post($service->getUrl(), $options);
        } catch (WebserviceRequestException $exception) {

            // Received an error response
            return $this->apiM->webservice()->response()->receive($client, $service, $request, $exception->getRequest()->getResponse());
        } catch (\GuzzleHttp\Exception\ClientException $exception) {

            // Received an error response
            return $this->apiM->webservice()->response()->receive($client, $service, $request, $exception->getResponse());
        }

        // Received response
        return $this->apiM->webservice()->response()->receive($client, $service, $request, $webserviceResponse);
    }
}
