<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace ExportProductGroup\Worker;

use ExportProductGroup\Api\Manager as ApiManager;
use ExportProductGroup\Config\Client;
use ExportProductGroup\Mapper\ProduktGruppe as ObjectMapper;
use ExportProductGroup\Model\Request\Objects\Update\Item;
use ExportProductGroup\Model\Request\Objects\Update\Request;
use ExportProductGroup\ValueMapper\MapperInterface as ValueMapper;
use Pimcore\Model\Object\Fieldcollection;
use Pimcore\Model\Object\ProduktGruppe;

/**
 * Update-Worker
 *
 * @package ExportProductGroup\Worker
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Update
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {

        $this->apiM = $apiM;
    }

    /**
     * @param \Zend_EventManager_Event $event
     *
     * @throws \ExportProductGroup\Config\InvalidConfigException
     */
    public function run(\Zend_EventManager_Event $event)
    {
        $object = $event->getTarget();

        // Object not published
        if (!$object->isPublished()) {
            $this->apiM->logger()->debug(sprintf('Object "%s" (ID %d) not exported: not published', $object->getKey(), $object->getId()));
            return;
        }

        // STYLEFILE-5486 : Dont' run export if only save new version is triggered within Pimcore
        if (true === $event->getParam('saveVersionOnly')) {
            return;
        }

        // Initialize object mapper
        $mapper = $this->apiM->mapper()->produktGruppe($object);

        foreach ($this->apiM->clients()->active() as $client) {

            // Prepare request
            $modelRequest = $this->apiM->model()->objectsUpdate()->request();

            // Prepare data
            $this->update($client, $modelRequest, $mapper);

            $this->apiM->logger()->debug(sprintf('Object "%s" (ID %d) added to update request',
                $object->getKey(),
                $object->getId()));

            // Send request
            $service = $this->apiM->clients()->service($client, 'Objects::Update');
            $this->apiM->webservice()->request()->send($modelRequest, $client, $service);
        }
    }

    /**
     * Update datasheet
     *
     * Prepare each attribute in each language even if it is an attribute
     * which is language independent. This makes live easier on the other site.
     *
     * @param Client       $client
     * @param Request      $request
     * @param ObjectMapper $objectMapper
     */
    private function update(Client $client, Request $request, ObjectMapper $objectMapper)
    {

        $item = $this->apiM->model()->objectsUpdate()->item($request);
        $item->setProductId($objectMapper->getObject()->getKey());

        // Cycle through configured languages
        foreach ($this->apiM->pimcore()->config()->getLanguages() as $language) {

            // Add language
            $item->addDocumentLanguage($language);

            // Switch language
            $this->apiM->translate()->translator()->setLocale($language);

            // Add properties
            foreach ($objectMapper->getObjectProperties() as $propertyName) {

                if ('localizedfields' == $propertyName) {

                    // Localized fields
                    foreach ($objectMapper->getLocalizedValues($language) as $key => $value) {
                        if (!empty($value)) {
                            $this->addLocalizedValue($objectMapper, $item, $key, $value, $language);
                        }
                    }

                } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Object\Fieldcollection')) {

                    // Field collection
                    $this->addFieldcollectionValues($objectMapper, $item, $objectMapper->getValue($propertyName), $language);

                } else {

                    // Plain value
                    $this->addValue($client, $objectMapper, $item, $propertyName, $language);

                }
            }
        }

    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param string       $key
     * @param string       $value
     * @param string       $language
     */
    private function addLocalizedValue(ObjectMapper $objectMapper, Item $item, $key, $value, $language)
    {

        $shopGroup = $this->determineShopGroup($key);

        $nameFieldList = $this->apiM->model()->objectsUpdate()->nameFieldList($item)
            ->setName($key)
            ->setLanguage($language)
            ->setLabelId($objectMapper->mapNameId($key))
            ->setLabel($this->apiM->translate()->translator()->trans($objectMapper->mapNameId($key)))
            ->setValue($value);

        if ($shopGroup != '') {
            $nameFieldList->setShopGroup($shopGroup);
        }
    }

    /**
     * assign and flag shop specific properties (stylefile, bigtree)
     *
     * @param string $fieldName
     *
     * @return string
     */
    private function determineShopGroup($fieldName)
    {

        if (strpos($fieldName, 'Stylefile') !== false) {

            return 'STYLEF';
        }


        if (strpos($fieldName, 'Bigtree') !== false) {

            return 'BIGT';
        }

        return '';
    }

    /**
     * @param ObjectMapper    $objectMapper
     * @param Item            $item
     * @param Fieldcollection $fieldCollection
     * @param string          $language
     */
    private function addFieldcollectionValues(ObjectMapper $objectMapper, Item $item, Fieldcollection $fieldCollection, $language)
    {

        $translator = $this->apiM->translate()->translator();

        $shopGroup = $this->determineShopGroup($fieldCollection->getFieldname());

        // Add field collection
        foreach ($fieldCollection->getItems() as $fieldCollectionData) {

            // Field collection mapper
            $fieldCollectionMapper = $this->apiM->mapper()->fieldCollection($fieldCollectionData);

            // Field collection data - item
            foreach ($fieldCollectionMapper->getValues() as $key => $values) {
                $nameFieldList = $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                    ->setName($key)
                    ->setLanguage($language)
                    ->setLabelId($objectMapper->mapNameId($key))
                    ->setLabel($translator->trans($objectMapper->mapNameId($key)));

                if ($shopGroup != '') {
                    $nameFieldList->setShopGroup($shopGroup);
                }

                // Each item in field collection data may have multiple values selected
                foreach ($values as $value) {
                    $this->apiM->model()->objectsUpdate()->valueItem($nameFieldList)
                        ->setValueId($value)
                        ->setValue($translator->trans($value));
                }
            }
        }
    }

    /**
     * @param Client       $client
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param string       $propertyName
     * @param string       $language
     */
    private function addValue(Client $client, ObjectMapper $objectMapper, Item $item, $propertyName, $language)
    {

        $translator = $this->apiM->translate()->translator();
        $propertyMapped = false;

        foreach ($this->apiM->pluginConfig()->getObjects()->getProduktGruppe()->getProperties() as $property) {

            // Property does not match
            if ($property->getId() != $propertyName) {
                continue;
            }

            // Ignore property
            if ($property->isDisabled()) {
                return;
            }

            // Map value
            $value = trim($this->valuemapper($client, $property->getMapper())->map($objectMapper, $property, $language));

            // Skip empty value
            if (!strlen($value)) {
                continue;
            }

            // Add node
            $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($property->getLabelId())
                ->setLabel($translator->trans($property->getLabel()))
                ->setValue($translator->trans($value));

            // Mark property as mapped
            $propertyMapped = true;
        }

        // Passthrough not mapped value
        if (!$propertyMapped && strlen($value = trim($objectMapper->getValue($propertyName)))) {
            $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($propertyName)
                ->setLabel($translator->trans($objectMapper->mapNameId($propertyName)))
                ->setValue($translator->trans($value));
        }
    }

    /**
     * @param Client $client
     * @param        $mapper
     *
     * @return ValueMapper
     */
    private function valueMapper(Client $client, $mapper)
    {

        return $this->apiM->valuemapper()->$mapper($client);
    }
}
