<?php
/**
 * Short description for file
 *
 * Long description for file (if any)...
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       www.life-style.de
 */

namespace Export2Swoox\Mapper\Stylefile;

use Export2Swoox\Mapper\MapperInterface;
use Export2Swoox\Plugin;
use Pimcore\Model\Element\ValidationException;
use Pimcore\Model\Object\ListingContent;
use Swoox\Api\Api\Manager;
use Swoox\Api\Model\Request\Delete\Swoox\Item as DeleteItem;
use Swoox\Api\Model\Request\Index\Swoox\Item as IndexItem;

/**
 * Class ListingContentMapper
 *
 * @package Export2Swoox\Mapper
 */
class ListingContentMapper implements MapperInterface
{
    private $itemData = [];
    const CONTENT = 'content';
    const INDEX = 'index';

    /**
     * @param      $data ListingContent
     * @param bool $index
     *
     * @return array
     */
    public function generateOutput($data, $index, $event)
    {
        $pluginConfig = Plugin::getPluginConfig();

        foreach ($pluginConfig['export2swoox']['pathconfig'] as $key => $pathConfig) {
            if (strpos($data->getFullPath(), $key) !== false) {
                $shopSuffix = $pathConfig['suffix'];
                $cdnPrefix = $pathConfig['cdn'];
                $cdnFallbackPrefix = $pathConfig['cdnFallback'];
            }
        }

        $manager = new Manager();
        $config = \Pimcore\Config::getSystemConfig();

        $languages = explode(',', $config->general->validLanguages);

        foreach ($languages as $language) {
            $objectId = 'lc-' . $language . $shopSuffix . '-' . $data->getId();
            if ($index) {
                $item = new IndexItem();
                $item->setDocumentLanguage($language);
                $item->setDocumentUniqueKey($objectId);
                $item->setDocumentTitle($data->getFullPath());
                $item->setDocumentPrefix(1);
                $item->setDocumentURL('');
                $item->setDocumentRanking(1);
                $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('PRODUCTTYPE')->setValue('LISTINGCONTENT'));
                $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('A_SYS_ID')->setValue($objectId));
                $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('LANG')->setValue($language));
                $hashesNode = $manager->model()->request()->index()->nameFieldList()->setName('LISTINGCONTENTHASH');

                $hashes = explode_and_trim("\n",
                    $data->getListingContentHashes()); //todo: more validation and sanitization

                if (empty($hashes)) {
                    throw new ValidationException('You need to set some hashes for listingContent');
                }

                foreach ($hashes as $hash) {
                    $hashValueItem = $manager->model()->request()->index()->valueItem();
                    $hashValueItem->setValue(explode('###', $hash)[0]);
                    $hashesNode->addValues($hashValueItem);
                }
                $item->addDocumentFields($hashesNode);

                $listingContentOutput = [];

                if (method_exists($data,'getListingContentZeilenTypen')
                    && method_exists($data->getListingContentZeilenTypen(),'getItems')
                ) {
                    $listingContentItems = $data->getListingContentZeilenTypen()->getItems();
                    $listingContentOutput = $this->collectDataFromListingContentLines($data, $listingContentItems,
                        $language, $cdnPrefix, $cdnFallbackPrefix, $listingContentOutput);
                }


                if (sizeof($listingContentOutput) == 0
                    && sizeof($data->getListingContentSeoTitle($language)) == 0
                    && sizeof($data->getListingContentHeadlineTitle($language)) == 0
                    && sizeof($data->getListingContentSeoDescription($language)) == 0
                    && sizeof($data->getListingContentCopyText($language)) == 0
                ) {
                    // there is no data entered for that specific language in both SEO- and ListingContent Data ->
                    // set unpublished
                    $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()
                        ->setName('PUBLISHED')->setValue(false));
                    $this->itemData[$language . $shopSuffix][] = $item;
                    continue;
                } else {
                    //alternative and fast way to get a flat view of a pimcore object
                    //$parsedObject = \Pimcore\Model\Webservice\Data\Mapper::map($data, "\\Pimcore\\Model\\Webservice\\Data\\Object\\Concrete\\Out", "out");
                    $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('LISTINGCONTENT')->setValue(json_encode($listingContentOutput)));
                    $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('PUBLISHED')->setValue($data->isPublished()));
                    $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('SEOTITLE')->setValue($data->getListingContentSeoTitle($language)));
                    $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('HEADLINETITLE')->setValue($data->getListingContentHeadlineTitle($language)));
                    $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('SEODESCRIPTION')->setValue($data->getListingContentSeoDescription($language)));
                    $item->addDocumentFields($manager->model()->request()->index()->nameFieldList()->setName('TEMPLATE')->setValue($data->getListingContentCopyText($language)));
                }
            } else {
                $item = new DeleteItem();
                $item->setDocumentUniqueKey($objectId);
            }
            //throw new ValidationException(print_r($item, true));
            $this->itemData[$language . $shopSuffix][] = $item;
        }
    }

    /**
     * @param $key
     * @return string
     */
    private function determineSlot($key)
    {
        if (strpos($key, 'Links') !== false) {
            return 'left';
        } elseif (strpos($key, 'Rechts') !== false) {
            return 'right';
        } else {
            return 'center';
        }
    }

    /**
     * @return array
     */
    public function getPayload()
    {
        return $this->itemData;
    }

    /**
     * @return bool
     */
    public function isMultiClientConfigured()
    {
        return true;
    }

    public function getEventWhitelist()
    {
        return ['object.postUpdate', 'object.preDelete'];
    }

    /**
     * @param $tempItem
     * @param $cdnPrefix
     * @param $cdnFallbackPrefix
     * @param $key
     * @param $tempContentItem
     * @param $readyToSend
     */
    private function rewritePictureDataAndCheckIfContentIsThere(
        &$tempItem,
        $cdnPrefix,
        $cdnFallbackPrefix,
        &$key,
        &$tempContentItem,
        &$readyToSend
    ) {
//goes through content and checks for pictures etc.
        foreach ($tempItem[self::CONTENT] as $key => $tempContentItem) {

            if ($tempContentItem instanceof \Pimcore\Model\Asset) {
                /**
                 * @var $tempContentItem \Pimcore\Model\Asset
                 */
                $fileModTime = $tempContentItem->getModificationDate();
                $pimcoreFilePath = $tempContentItem->getFullPath();

                $url = $pimcoreFilePath . '?m=' . $fileModTime;
                $tempItem[self::CONTENT][$key] = [
                    'cdn' => $cdnPrefix . $url,
                    'cdnFallback' => $cdnFallbackPrefix . $url
                ];
            }

            if (!is_null($tempContentItem) && strlen(trim($tempContentItem)) > 0) {
                //maybe there is a empty object with only nulls...
                $readyToSend = true;
            }
        }
    }

    /**
     * @param $data
     * @param $tempItem
     * @param $key
     * @param $tempContentItem
     */
    private function rearrangeContentSlots($data, &$tempItem, &$key, &$tempContentItem)
    {
        $contentArray = [];
        //iterate and order some properties
        foreach ($tempItem[self::CONTENT] as $key => $tempContentItem) {
            $contentArray[$this->determineSlot($key)][str_replace([
                'listingContent',
                'Links',
                'Rechts',
                'Mitte'
            ], '', $key)] = $tempContentItem;
        }
        $tempItem[self::CONTENT] = $contentArray;

        foreach ($tempItem[self::CONTENT] as $key => $itemSlot) {
            //Add global data to each slot, because in frontend a mixture of slots is possible
            $tempItem[self::CONTENT][$key]['published'] = $data->isPublished();
            $tempItem[self::CONTENT][$key]['source'] = $data->getFullPath();
        }
    }

    /**
     * @param $data
     * @param $listingContentItems
     * @param $language
     * @param $cdnPrefix
     * @param $cdnFallbackPrefix
     * @param $listingContentOutput
     * @return mixed
     */
    private function collectDataFromListingContentLines(
        $data,
        $listingContentItems,
        $language,
        $cdnPrefix,
        $cdnFallbackPrefix,
        $listingContentOutput
    ) {
        $defaultLanguage = \Pimcore\Tool::getDefaultLanguage();
        foreach ($listingContentItems as $contentItem) {
            $readyToSend = false;
            $tempItem = [];

            if (strpos($contentItem->getType(), 'Scheduled') !== false) {
                $tempItem['scheduled'] = true;
                $tempItem['type'] = str_replace('Scheduled', '', $contentItem->getType());
            } else {
                $tempItem['scheduled'] = false;
                $tempItem['type'] = $contentItem->getType();
            }

            if (isset($contentItem->listingContentMobile) && $contentItem->listingContentMobile) {
                $tempItem['type'] = $tempItem['type'] . 'Mobile';
            }

            $tempItem[self::INDEX] = $contentItem->getListingContentZeilenNummer();

            // if listingContentGlobalDefault is set to true the listing content of the default language is spread to
            // all other languages
            if (isset($contentItem->listingContentGlobalDefault) && $contentItem->listingContentGlobalDefault) {
                $tempItem[self::CONTENT] = $contentItem->getLocalizedfields()->items[$defaultLanguage];
            } else {
                $tempItem[self::CONTENT] = $contentItem->getLocalizedfields()->items[$language];
            }
            
            $this->rewritePictureDataAndCheckIfContentIsThere($tempItem, $cdnPrefix, $cdnFallbackPrefix, $key,
                $tempContentItem, $readyToSend);

            if (!$readyToSend || is_null($tempItem[self::INDEX])) {
                //todo: more validation
                //There is something wrong with this line...just skip to the next Line
                continue;
            }

            $fieldDefs = $contentItem->getDefinition()->getFieldDefinitions();

            //just get schedule properties and add this also to the content
            foreach ($fieldDefs as $fieldDef) {
                if ($fieldDef->getName() != 'listingContentZeilenNummer' && $fieldDef->getName() != 'localizedfields' && $fieldDef->getFieldtype() == 'datetime') {
                    $tempItem[self::CONTENT][$fieldDef->getName()] = $contentItem->{$fieldDef->getName()}->getTimestamp();
                }
            }

            $this->rearrangeContentSlots($data, $tempItem, $key, $tempContentItem);

            $listingContentOutput[$tempItem[self::INDEX]][$tempItem['type']][] = $tempItem;

        }
        return $listingContentOutput;
    }
}
