<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Export2Swoox\Webservice;

use Export2Swoox\Mapper\MapperInterface;
use GuzzleHttp\Client;
use JMS\Serializer\SerializerBuilder;
use Pimcore\Log\ApplicationLogger;
use Swoox\Api\Api\Manager;

/**
 * Webservice-Request
 *
 * todo: split in two instances and get rid of if / else of indextype
 * @package SwooxModel\Webservice
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Request
{

    /**
     * @param      $mapper MapperInterface
     * @param bool $indexRequest
     */
    public function prepareAndSend($mapper, $indexRequest)
    {
        $config = new Config();
        $payload = $mapper->getPayload();

        //je nachdem, ob Items in unterschiedliche Swoox-Clients verteilt werden sollen...
        if ($mapper->isMultiClientConfigured()) {
            foreach ($payload as $key => $items) {
                $request = $this->addItemsToSwooxRequest($items, $indexRequest);
                $client = $config->getClientId($key);
                if ($client) {
                    //Falls kein Client für jeweiliges Land konfiguriert ist -> Skip
                    $this->send($request, $client, $config, $indexRequest);
                }
            }
            // run deleted checks after delete requests finished
            if (!$indexRequest) {
                // give es nodes some time to sync, so take a short break first...
                sleep(1);
                foreach ($payload as $key => $items) {
                    $request = $this->addItemsToSwooxRequest($items, $indexRequest);
                    $clientId = $config->getClientId($key);
                    if ($clientId) {
                        $resultCount = $this->getResultCountFromSwoox($request, $clientId, $config);
                        $this->validateDeletion($resultCount);
                    }
                }
            }
        } else {
            $request = $this->addItemsToSwooxRequest($payload, $indexRequest);
            $this->send($request, $config->getClientId(), $config, $indexRequest);
        }
    }

    /**
     * @param $request
     * @param $clientId
     * @param $config Config
     * @param $indexRequest
     */
    private function send($request, $clientId, $config, $indexRequest)
    {

        $serializer = SerializerBuilder::create()->build();

        $options = [
            'headers' => [
                'Accept' => $config->getAcceptContentType(),
                'Content-Type' => $config->getContentType(),
                'clientid' => $clientId,
            ],
            'body' => $serializer->serialize($request, $config->getFormat()),
        ];

        ApplicationLogger::getInstance("Export2Swoox", true)->debug('Request:' . json_encode($options));
        $client = new Client();
        if ($indexRequest) {
            $webserviceResponse = $client->put($config->getUpdateUrl(), $options);
        } else {
            $webserviceResponse = $client->delete($config->getDeleteUrl(), $options);
        }

        ApplicationLogger::getInstance("Export2Swoox",
            true)->debug('Response:' . $webserviceResponse->getBody()->getContents());
        //Wenn wenn es kracht, soll der Fehler ruhig ans Frontend
    }

    /**
     * @param $swooxModelManager
     * @param $payload
     *$config->getSearchUrl()
     * @return mixed
     */
    private function addItemsToSwooxRequest($payload, $indexRequest)
    {
        $swooxModelManager = new Manager();

        if ($indexRequest) {
            $request = $swooxModelManager->model()->request()->index()->swoox();
        } else {
            $request = $swooxModelManager->model()->request()->delete()->swoox();
        }

        foreach ($payload as $items) {
            $request->addItem($items);
        }

        return $request;
    }

    /**
     * @param $request
     * @param $clientId
     * @param $config
     *
     * @return integer
     * @throws \Exception
     */
    private function getResultCountFromSwoox($request, $clientId, $config)
    {
        $client = new Client();
        $documentUniqueKey = $request->getItems()[0]->getDocumentUniqueKey();
        $query = [
            'query' => [
                'clientid' => $clientId,
                '1' => $documentUniqueKey,
                'nocache' => 1,
            ],
        ];
        $searchResponse = $client->get($config->getSearchUrl(), $query);

        return intval(json_decode($searchResponse->getBody()->getContents())->ResultsCount);
    }

    /**
     * @param integer $resultCount
     *
     * @throws \Exception
     */
    private function validateDeletion($resultCount)
    {
        if (0 != $resultCount) {
            ApplicationLogger::getInstance("Export2Swoox",
                true)->debug('Problem: Item found after delete');
            throw new \Exception('Item found after delete');
        }
    }
}
