<?php

/**
 * PimCore Export Translations Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Publikat\Pimcore\ExportTranslationsBundle\Webservice;

use Publikat\Pimcore\ExportTranslationsBundle\Api\Manager as ApiManager;
use Publikat\Pimcore\ExportTranslationsBundle\Config\Client as ClientConfig;
use Publikat\Pimcore\ExportTranslationsBundle\Config\Service as ServiceConfig;
use Publikat\Pimcore\ExportTranslationsBundle\Model\Request\RequestInterface;
use Publikat\Pimcore\ExportTranslationsBundle\Model\Response\ResponseInterface;
use Publikat\Pimcore\ExportTranslationsBundle\Model\Response\Errors\Errors as ErrorResponse;
use GuzzleHttp\Psr7\Response as WebserviceResponse;

/**
 * Webservice-Response
 *
 * @package Publikat\Pimcore\ExportTranslationsBundle\Webservice
 * @author Oliver Friedrich <of@life-style.de>
 */
class Response
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
    }

    /**
     * Receive success response
     *
     * @param ClientConfig $client
     * @param ServiceConfig $service
     * @param RequestInterface $request
     * @param WebserviceResponse $webserviceResponse
     * @return ResponseInterface
     * @throws InvalidResponseException
     */
    public function receive(ClientConfig $client, ServiceConfig $service, RequestInterface $request, WebserviceResponse $webserviceResponse)
    {
        // Received an error
        if (200 != $webserviceResponse->getStatusCode()) {
            return $this->error($client, $service, $webserviceResponse);
        }

        // Invalid content type
        if (!$this->validateContentType($service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0])) {
            $this->apiM->logger()->debug('Response ' . $webserviceResponse->getBody()->getContents());
            throw new InvalidResponseException(sprintf('Error while updating translation: ' . PHP_EOL . 'Webservice-response should be type of %s but is %s', $service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0]));
        }

        // Receive body
        $body = $webserviceResponse->getBody()->getContents();

        $this->apiM->logger()->info(sprintf('Translation export: received success response for client "%s"', $client->getId()));
        $this->apiM->logger()->debug($body);

        return $this->apiM->serializer()->deserialize($body, $request->getResponseType(), $service->getFormat());
    }

    /**
     * Receive error response
     *
     * @param ClientConfig $client
     * @param ServiceConfig $service
     * @param WebserviceResponse $webserviceResponse
     * @throws InvalidResponseException
     */
    private function error(ClientConfig $client, ServiceConfig $service, WebserviceResponse $webserviceResponse)
    {
        // Receive body
        $body = $webserviceResponse->getBody()->getContents();

        $this->apiM->logger()->crit(sprintf('Translation export: received error response status(%d) for client "%s"', $webserviceResponse->getStatusCode(), $client->getId()));
        $this->apiM->logger()->err($body);

        // Invalid content type
        if (!$this->validateContentType($service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0])) {
            throw new InvalidResponseException(sprintf('Error while updating translation: ' . PHP_EOL . 'Webservice-response should be %s but is %s', $service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0]));
        }

        $response = $this->apiM->serializer()->deserialize($body, 'Publikat\Pimcore\ExportTranslationsBundle\Model\Response\Errors\Errors', $service->getFormat());
        throw new InvalidResponseException('Error while updating objects: ' . PHP_EOL . implode(PHP_EOL, $this->getErrors($response)));
    }

    /**
     * Returns true, if content types matches
     *
     * @param string $contentTypeA
     * @param string $contentTypeB
     * @return bool
     */
    private function validateContentType($contentTypeA, $contentTypeB)
    {
        return 0 === strcasecmp(preg_replace('/;.*$/', '', $contentTypeA), preg_replace('/;.*$/', '', $contentTypeB));
    }

    /**
     * @param ErrorResponse $response
     * @return array
     */
    private function getErrors(ErrorResponse $response)
    {
        $errors = array();
        foreach ($response->getError() as $error) {

            // Add error message
            $errors[] = $error->getMessage();

            // Add additional information if in debug mode
            if ($this->apiM->pimcore()->config()->isDebugMode()) {
                if ($error->getCode()) {
                    $errors[] = 'Error code: ' . $error->getCode();
                }
                if ($error->getDebug()) {
                    $errors[] = 'Debug message: ' . $error->getDebug();
                }
                if ($error->getException()) {
                    $errors[] = 'Exception message: ' . $error->getException();
                }
            }
        }

        if (!count($errors)) {
            $errors[] = 'An unknown error has occured in webservice-communication.';
        }
        return $errors;
    }
}