<?php

/**
 * PimCore Export Translations Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Publikat\Pimcore\ExportTranslationsBundle\Worker;

use Publikat\Pimcore\ExportTranslationsBundle\Api\Manager as ApiManager;
use Publikat\Pimcore\ExportTranslationsBundle\Model\Request\Translations\Update\Request as UpdateRequest;
use Publikat\Pimcore\ExportTranslationsBundle\Model\Request\Translations\Delete\Request as DeleteRequest;
use Pimcore\Model\Object\AttributUebersetzung\Listing as AttributUebersetzungListing;
use Pimcore\Model\Object\AttributUebersetzung;

/**
 * Export Worker - Full Export of Translations
 *
 * @package Publikat\Pimcore\ExportTranslationsBundle\Worker
 * @author Oliver Friedrich <of@life-style.de>
 */
class Export
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
    }

    /**
     * @return array
     * @throws \Publikat\Pimcore\ExportTranslationsBundle\Config\InvalidConfigException
     */
    public function run()
    {
        $this->apiM->logger()->info('Starting full export of translations ...');

        // Prepare request
        $modelRequestUpdate = $this->apiM->model()->translationsUpdate()->request();
        $modelRequestDelete = $this->apiM->model()->translationsDelete()->request();

        // Get list of translations
        $translationsUpdateCount = 0;
        $translationsDeleteCount = 0;
        $translations = new AttributUebersetzungListing();
        foreach ($translations as $object) {

            // Delete unpublished translation
            if (!$object->isPublished()) {
                $this->delete($modelRequestDelete, $object);
                ++$translationsDeleteCount;

                $this->apiM->logger()->debug(sprintf('Translation "%s" (ID %d) added to delete request',
                    $object->getKey(),
                    $object->getId()));

                continue;
            }

            // Prepare update translation
            $this->update($modelRequestUpdate, $object);
            ++$translationsUpdateCount;

            $this->apiM->logger()->debug(sprintf('Translation "%s" (ID %d) added to update request',
                $object->getKey(),
                $object->getId()));
        }

        $this->apiM->logger()->info(sprintf('Prepared translations for delete %d / update %d ...', $translationsDeleteCount, $translationsUpdateCount));

        // Send request
        foreach ($this->apiM->clients()->active() as $client) {
            if ($translationsDeleteCount) {
                $this->apiM->webservice()->request()->send($modelRequestDelete, $client, $this->apiM->clients()->service($client, 'Translation::Delete'));
            }
            if ($translationsUpdateCount) {
                $this->apiM->webservice()->request()->send($modelRequestUpdate, $client, $this->apiM->clients()->service($client, 'Translation::Update'));
            }
        }

        return array($translationsUpdateCount, $translationsDeleteCount);
    }

    /**
     * @param UpdateRequest $request
     * @param AttributUebersetzung $object
     */
    private function update(UpdateRequest $request, AttributUebersetzung $object)
    {
        $item = $this->apiM->model()->translationsUpdate()->item($request);
        $item->setName($object->getKey());

        // Cycle through configured languages
        foreach ($this->apiM->pimcore()->config()->getLanguages() as $language) {

            // Skip empty values
            $value = $object->getAttributUebersetzungText($language);
            if (null === $value) {
                continue;
            }

            // Add translation
            $this->apiM->model()->translationsUpdate()->valueItem($item)
                ->setLanguage($language)
                ->setValue($value);
        }
    }

    /**
     * @param DeleteRequest $request
     * @param AttributUebersetzung $object
     */
    private function delete(DeleteRequest $request, AttributUebersetzung $object)
    {
        $this->apiM->model()->translationsDelete()->item($request)
            ->setName($object->getKey());
    }
}