<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace ImportArticle\Tests;

use ImportArticle\Worker\Import;
use Pimcore\Model\Object\ArtikelFamilie;
use Pimcore\Model\Object\ProduktGruppe;
use Publikat\Model\Xml\Item\Availability;
use Publikat\Model\Xml\Item\ShopData;

require_once dirname(__FILE__) . '/TestBase.php';

/***
 * Class WorkerTest
 *
 * @package ImportArticle\Tests
 */
class WorkerTest extends TestBase
{
    const PIMCORE_MODEL_OBJECT_PRODUKT_GRUPPE = '\Pimcore\Model\Object\ProduktGruppe';
    const PIMCORE_MODEL_OBJECT_DATA_LINK = '\Pimcore\Model\Object\Data\Link';
    const PIMCORE_MODEL_OBJECT_ABSTRACT_OBJECT = '\Pimcore\Model\Object\AbstractObject';

    /**
     * @var Import
     */
    private $instance = null;

    /**
     * init
     */
    const PRODUCT_GROUP_CODE = '41507';

    public function init()
    {
        $apiM = $this->apiM();
        $this->instance = new Import($apiM);
    }

    /**
     * This test will work with the local pimcore instance. If there is no valid $path there will be an exception.
     */
    public function testGetFolderTrue()
    {
        $this->init();
        $path = 'oberbekleidung';

        $object = $this->instance->getFolderArtikelFamilie($path);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_ABSTRACT_OBJECT, $object);

        $object = $this->instance->getFolderProduktGruppe($path);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_ABSTRACT_OBJECT, $object);
    }

    /**
     * @expectedException \Exception
     */
    public function testGetFolderArtikelFamilieFalse()
    {
        $this->init();
        $path = 'penunsier';
        $object = $this->instance->getFolderArtikelFamilie($path);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_ABSTRACT_OBJECT, $object);
    }

    /**
     * @expectedException \Exception
     */
    public function testGetFolderProduktGruppeFalse()
    {
        $this->init();
        $path = 'penunsier';
        $object = $this->instance->getFolderProduktGruppe($path);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_ABSTRACT_OBJECT, $object);
    }

    /**
     * testCategoryMappingShopTrue
     */
    public function testCategoryMappingShopTrue()
    {
        $this->init();
        /**
         * Oberbekleidung
         */
        $categoryId = 714;
        $object = $this->instance->categoryMappingShop($categoryId);

        $this->assertTrue(is_string($object));
        $this->assertTrue($object == 'Oberbekleidung');
    }

    /**
     * testCategoryMappingShopFalse
     */
    public function testCategoryMappingShopFalse()
    {
        $this->init();
        $categoryId = 71890123456;
        $object = $this->instance->categoryMappingShop($categoryId);

        $this->assertEquals('fehlendes-mapping', $object);
    }

    /**
     * This test takes some seconds because it will ask the local configured pimcore instance for the asked data
     *
     * testLoadArtikelFamilieTrue
     */
    public function testLoadArtikelFamilieTrue()
    {
        $this->init();
        $articleFamilyCode = 'KJ10B072';
        $path = 'oberbekleidung';
        $articleFamilyObject = '\Pimcore\Model\Object\ArtikelFamilie';

        $object = $this->instance->loadArtikelFamilie($articleFamilyCode, $path);
        /**
         * object already exists
         */
        $this->assertInstanceOf($articleFamilyObject, $object);
        $this->assertTrue(is_int($object->getCreationDate()));

        $articleFamilyCode = 'KJ10B072123123123123123';
        $object = $this->instance->loadArtikelFamilie($articleFamilyCode, $path);
        /**
         * object does not exist -> new instance
         */
        $this->assertInstanceOf($articleFamilyObject, $object);
        $this->assertFalse(is_int($object->getCreationDate()));
    }

    /**
     * This test takes some seconds because it will ask the local configured pimcore instance for the asked data
     *
     * testProduktGruppeTrue
     */
    public function testProduktGruppeTrue()
    {
        $this->init();
        $produktGruppenCode = self::PRODUCT_GROUP_CODE; // /artikel/unterbekleidung/produktgruppen/41507
        $path = 'unterbekleidung';

        $object = $this->instance->loadProduktGruppe($produktGruppenCode, $path);
        /**
         * object already exists
         */
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_PRODUKT_GRUPPE, $object);
        $this->assertTrue(is_int($object->getCreationDate()));

        $produktGruppenCode = '41507123123123123123';
        $object = $this->instance->loadProduktGruppe($produktGruppenCode, $path);
        /**
         * object does not exist -> new instance
         */
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_PRODUKT_GRUPPE, $object);
        $this->assertFalse(is_int($object->getCreationDate()));
    }

    // This test takes some seconds because it will ask the local configured pimcore instance for the asked data

    /**
     * testLoadObjectFalse
     */
    public function testLoadObjectFalse()
    {
        $this->init();
        $type = 'DummbatzToll';
        $object = $this->instance->loadObject(null, null, null, $type);
        $this->assertTrue($object == null);
    }

    /**
     * testbuildLinkArtikelFamilieTrue
     */
    public function testbuildLinkArtikelFamilieTrue()
    {

        $this->init();
        $articleFamilyCode = 'KJ10B072';
        $object = $this->instance->buildLinkArtikelFamilieBigTree($articleFamilyCode);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_DATA_LINK, $object);
        $this->assertTrue($object->getText() === '<a href="http://www.bigtree.de/abenteuer-und-outdoor-ausruestung/?keywords=' . $articleFamilyCode . '" target="_blank">zum Shop</a>');

        $object = $this->instance->buildLinkArtikelFamilieStylefile($articleFamilyCode);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_DATA_LINK, $object);
        $this->assertTrue($object->getText() === '<a href="http://www.stylefile.de/streetwear-sneaker-graffiti/?keywords=' . $articleFamilyCode . '" target="_blank">zum Shop</a>');
    }

    /**
     * testbuildLinkProduktGruppeTrue
     */
    public function testbuildLinkProduktGruppeTrue()
    {
        $this->init();
        $produktGruppenCode = self::PRODUCT_GROUP_CODE;
        $object = $this->instance->buildLinkProduktGruppeBigTree($produktGruppenCode);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_DATA_LINK, $object);
        $this->assertTrue($object->getText() === '<a href="http://www.bigtree.de/xxx-fid-' . $produktGruppenCode . '.html" target="_blank">zum Shop</a>');

        $object = $this->instance->buildLinkProduktGruppeStylefile($produktGruppenCode);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_DATA_LINK, $object);
        $this->assertTrue($object->getText() === '<a href="http://www.stylefile.de/xxx-fid-' . $produktGruppenCode . '.html" target="_blank">zum Shop</a>');
    }

    /**
     * testBuildLinkFalse
     */
    public function testBuildLinkFalse()
    {

        $this->init();
        $object = $this->instance->buildLink(null, 'artikelfamilien', null);
        $this->assertNull($object);

        $object = $this->instance->buildLink(null, null, 1);
        $this->assertNull($object);

        $object = $this->instance->buildLink(null, 'DummbatzToll', 1);
        $this->assertNull($object);
    }

    /**
     * testDeserializeXMLTrue
     */
    public function testDeserializeXMLTrue()
    {
        $this->init();
        $testXML = "<Item>
                        <ItemTransfer>
                             <Availability>
                                <ProductGroupCode><![CDATA[41507]]></ProductGroupCode>
                                <RealQty>12</RealQty>
                                <ShippingState>11</ShippingState>
                                <Item>
                                    <ShopGroup><![CDATA[STYLEF]]></ShopGroup>
                                </Item>
                             </Availability>
                                <ShopData>
                                    <ActivationDate><![CDATA[1970-01-01 01:00:00]]></ActivationDate>
                                    <ArticleFamilyCode><![CDATA[KH49611]]></ArticleFamilyCode>
                                    <ArticleName><![CDATA[Adidas Europa TP Hose]]></ArticleName>
                                    <CareLabeling><![CDATA[Material: 52.00% Baumwolle<br>Material: 48.00% Polyester]]></CareLabeling>
                                    <CategoryId>130</CategoryId>
                                    <CategoryName><![CDATA[Training Pants]]></CategoryName>
                                    <Color><![CDATA[rot blau gelb]]></Color>
                                    <FamilyLinkId><![CDATA[41431]]></FamilyLinkId>
                                    <FullUpdate><![CDATA[26.02.2015 13:59:16<br>29.09.2014 10:20:14<br>28.08.2014 19:55:13<br>28.08.2014 18:38:29<br>28.07.2014 09:10:12<br>01.07.2014 13:14:29<br>13.05.2014 13:07:08<br>04.04.2014 05:07:40<br>13.03.2014 17:36:56<br>14.02.2014 20:00:44]]></FullUpdate>
                                    <PartialUpdate><![CDATA[21.07.2014 04:03:44<br>21.07.2014 04:03:44<br>21.07.2014 04:03:44<br>21.07.2014 04:03:43<br>21.07.2014 04:03:43<br>21.07.2014 04:03:42<br>22.08.2013 16:41:35<br>22.08.2013 16:41:34<br>22.08.2013 16:41:34<br>22.08.2013 16:41:34]]></PartialUpdate>
                                    <Price>64.9</Price>
                                    <ShopAssignment><![CDATA[STF-AMA-DE<br>STF-AMA-FR<br>STF-COM<br>STF-DE<br>STF-ES<br>STF-FR<br>STF-IT<br>STF-NL]]></ShopAssignment>
                                    <VendorItemNo><![CDATA[F78141]]></VendorItemNo>
                                </ShopData>
                            </ItemTransfer>
                        </Item>";
        $object = $this->instance->deserializeXML($testXML);
        $this->assertInstanceOf('\ImportArticle\Model\Request\Item', $object);
    }

    /**
     * @expectedException \Exception
     */
    public function testDeserializeXMLFalse()
    {
        $this->init();
        $testXML = "<Item> Fucked up XML";
        $object = $this->instance->deserializeXML($testXML);
        $this->assertInstanceOf('\ImportArticle\Model\Request\Item', $object);
    }

    /**
     * testSetDataProduktGruppeTrue
     */
    public function testSetDataProduktGruppeTrue()
    {
        $this->init();
        $shopData = new ShopData();
        $availability = new Availability();
        $productGroup = new ProduktGruppe();

        $availability->setShippingState(21);
        $object = $this->instance->setDataProduktGruppe($productGroup, $shopData, $availability);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_PRODUKT_GRUPPE, $object);

        $availability->setShippingState(22);
        $object = $this->instance->setDataProduktGruppe($productGroup, $shopData, $availability);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_PRODUKT_GRUPPE, $object);

        $availability->setShippingState(11);
        $object = $this->instance->setDataProduktGruppe($productGroup, $shopData, $availability);
        $this->assertInstanceOf(static::PIMCORE_MODEL_OBJECT_PRODUKT_GRUPPE, $object);
    }

    /**
     * testDataArtikelFamilieTrue
     */
    public function testDataArtikelFamilieTrue()
    {
        $this->init();
        $shopData = new ShopData();
        $availability = new Availability();
        $articleFamily = new ArtikelFamilie();
        $productGroup = new ProduktGruppe();
        $object = $this->instance->setDataArtikelFamilie($articleFamily, $productGroup, $shopData, $availability);
        $this->assertInstanceOf('\Pimcore\Model\Object\ArtikelFamilie', $object);
    }

    /**
     * testRunTrue
     */
    public function testRunTrue()
    {
        $this->init();
        $testXML = "<Item>
                        <ItemTransfer>
                             <Availability>
                                <ProductGroupCode><![CDATA[41507]]></ProductGroupCode>
                                <RealQty>12</RealQty>
                                <ShippingState>11</ShippingState>
                                <Item>
                                    <ShopGroup><![CDATA[STYLEF]]></ShopGroup>
                                </Item>
                             </Availability>
                                <ShopData>
                                    <ActivationDate><![CDATA[1970-01-01 01:00:00]]></ActivationDate>
                                    <ArticleFamilyCode><![CDATA[KH49611]]></ArticleFamilyCode>
                                    <ArticleName><![CDATA[Adidas Europa TP Hose]]></ArticleName>
                                    <CareLabeling><![CDATA[Material: 52.00% Baumwolle<br>Material: 48.00% Polyester]]></CareLabeling>
                                    <CategoryId>130</CategoryId>
                                    <CategoryName><![CDATA[Training Pants]]></CategoryName>
                                    <Color><![CDATA[rot blau gelb]]></Color>
                                    <FamilyLinkId><![CDATA[41431]]></FamilyLinkId>
                                    <FullUpdate><![CDATA[26.02.2015 13:59:16<br>29.09.2014 10:20:14<br>28.08.2014 19:55:13<br>28.08.2014 18:38:29<br>28.07.2014 09:10:12<br>01.07.2014 13:14:29<br>13.05.2014 13:07:08<br>04.04.2014 05:07:40<br>13.03.2014 17:36:56<br>14.02.2014 20:00:44]]></FullUpdate>
                                    <PartialUpdate><![CDATA[21.07.2014 04:03:44<br>21.07.2014 04:03:44<br>21.07.2014 04:03:44<br>21.07.2014 04:03:43<br>21.07.2014 04:03:43<br>21.07.2014 04:03:42<br>22.08.2013 16:41:35<br>22.08.2013 16:41:34<br>22.08.2013 16:41:34<br>22.08.2013 16:41:34]]></PartialUpdate>
                                    <Price>64.9</Price>
                                    <ShopAssignment><![CDATA[STF-AMA-DE<br>STF-AMA-FR<br>STF-COM<br>STF-DE<br>STF-ES<br>STF-FR<br>STF-IT<br>STF-NL]]></ShopAssignment>
                                    <VendorItemNo><![CDATA[F78141]]></VendorItemNo>
                                </ShopData>
                            </ItemTransfer>
                        </Item>";
        $object = $this->instance->run($testXML);
        $objectClean = json_decode($object, true);
        $this->assertArrayHasKey(self::PRODUCT_GROUP_CODE, $objectClean);
        $this->assertTrue($objectClean[self::PRODUCT_GROUP_CODE]['success']);
        // now check if object has all xml data

        $object = $this->instance->loadProduktGruppe(self::PRODUCT_GROUP_CODE, 'unterbekleidung');

        $this->assertEquals('STYLEF', $object->getDisplayShopGruppe());
        $this->assertEquals('12', $object->getDisplayLagerbestand());
    }

    /**
     * @expectedException \Exception
     */
    public function testRunFalse()
    {
        $this->init();
        $testXML = "<Item> Fucked up XML";
        $object = $this->instance->run($testXML);
        $this->assertTrue($object === true);
    }
}
