<?php

/**
 * PimCore Article Import Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ImportArticle\Worker;

use ImportArticle\Api\Manager as ApiManager;
use Pimcore\Date;
use Pimcore\File;
use Pimcore\Model\Object\ArtikelFamilie;
use Pimcore\Model\Object\Data\Link;
use Pimcore\Model\Object\Folder;
use Pimcore\Model\Object\ProduktGruppe;
use Publikat\Model\Xml\Item\Availability;
use Publikat\Model\Xml\Item\ShopData;

/**
 * Class Import
 *
 * @package ImportArticle\Worker
 */
class Import
{
    const ARTIKELFAMILIEN = 'artikelfamilien';
    const PRODUKTGRUPPEN = 'produktgruppen';

    const DIVERSE = 'diverse';
    const UNTERBEKLEIDUNG = 'Unterbekleidung';
    const SCHUHE = 'Schuhe';
    const OBERBEKLEIDUNG = 'Oberbekleidung';
    const CAPS_MUETZEN = 'caps-muetzen';
    const TASCHEN_ALLGEMEIN = 'taschen-allgemein';
    const DOSEN = 'Dosen';
    const TEXTILIEN_ALLGEMEIN = 'textilien-allgemein';
    const RUCKSAECKE = 'Rucksaecke';
    const WERKZEUG_REGENSCHIRME = 'werkzeug-regenschirme';
    const WASSERBEHAELTER_KUECHENUTENSILIEN = 'wasserbehaelter-kuechenutensilien';
    const PFLEGE_1_H = 'pflege-1.-h.';
    const NAHRUNG = 'Nahrung';
    const KOCHER = 'Kocher';
    const SCHLAFSAECKE = 'Schlafsaecke';
    const LEUCHTMITTEL = 'Leuchtmittel';
    const ORIENTIERUNG_GPS = 'orientierung-gps';
    const AKKUS_LADEGERAETE = 'Akkus-Ladegeraete';
    const EISENWAREN = 'eisenwaren';
    const STOECKE = 'Stoecke';
    const MATTEN_BETTEN = 'matten-betten';
    const MOEBEL = 'Moebel';
    const ACCESSOIRES = 'accessoires';
    const BUECHER_MEDIEN = 'buecher-medien';
    const MARKER_REFILL = 'marker-refill';
    const ZELT = 'Zelt';

    /**
     * @var string
     */
    private $mappingShopDefault = 'fehlendes-mapping';

    /**
     * @var array
     */
    private $mappingShop = [
        7 => self::DIVERSE,
        13 => self::UNTERBEKLEIDUNG,
        15 => self::SCHUHE,
        16 => self::OBERBEKLEIDUNG,
        17 => self::OBERBEKLEIDUNG,
        28 => self::CAPS_MUETZEN,
        29 => self::OBERBEKLEIDUNG,
        30 => self::OBERBEKLEIDUNG,
        31 => self::OBERBEKLEIDUNG,
        33 => self::OBERBEKLEIDUNG,
        34 => self::UNTERBEKLEIDUNG,
        35 => self::TASCHEN_ALLGEMEIN,
        36 => self::DIVERSE,
        37 => self::TASCHEN_ALLGEMEIN,
        38 => self::CAPS_MUETZEN,
        41 => self::OBERBEKLEIDUNG,
        42 => self::OBERBEKLEIDUNG,
        44 => self::OBERBEKLEIDUNG,
        47 => self::OBERBEKLEIDUNG,
        48 => self::UNTERBEKLEIDUNG,
        50 => self::OBERBEKLEIDUNG,
        51 => self::UNTERBEKLEIDUNG,
        52 => self::DIVERSE,
        57 => self::DOSEN,
        59 => self::DOSEN,
        60 => self::DIVERSE,
        61 => self::DIVERSE,
        62 => self::DIVERSE,
        63 => self::DIVERSE,
        64 => self::DIVERSE,
        65 => self::DIVERSE,
        66 => self::DIVERSE,
        67 => self::DIVERSE,
        68 => self::DIVERSE,
        70 => self::DIVERSE,
        71 => self::DIVERSE,
        72 => self::DIVERSE,
        73 => self::DOSEN,
        74 => self::DOSEN,
        75 => self::DOSEN,
        76 => self::DOSEN,
        77 => self::DOSEN,
        78 => self::DOSEN,
        79 => self::DOSEN,
        80 => self::DOSEN,
        82 => self::DIVERSE,
        99 => self::DOSEN,
        101 => self::OBERBEKLEIDUNG,
        103 => self::DIVERSE,
        104 => self::DIVERSE,
        105 => self::DIVERSE,
        106 => self::UNTERBEKLEIDUNG,
        107 => self::CAPS_MUETZEN,
        108 => self::TEXTILIEN_ALLGEMEIN,
        109 => self::UNTERBEKLEIDUNG,
        110 => self::SCHUHE,
        111 => self::UNTERBEKLEIDUNG,
        114 => self::CAPS_MUETZEN,
        115 => self::OBERBEKLEIDUNG,
        116 => self::CAPS_MUETZEN,
        117 => self::UNTERBEKLEIDUNG,
        118 => self::UNTERBEKLEIDUNG,
        119 => self::RUCKSAECKE,
        120 => self::SCHUHE,
        121 => self::SCHUHE,
        122 => self::SCHUHE,
        123 => self::SCHUHE,
        125 => self::TEXTILIEN_ALLGEMEIN,
        126 => 'Sonnenbrillen',
        128 => self::SCHUHE,
        129 => self::OBERBEKLEIDUNG,
        130 => self::UNTERBEKLEIDUNG,
        132 => self::DIVERSE,
        134 => self::DIVERSE,
        135 => self::SCHUHE,
        136 => self::SCHUHE,
        137 => self::DOSEN,
        138 => self::DOSEN,
        139 => self::DOSEN,
        140 => self::DOSEN,
        141 => self::DOSEN,
        142 => self::DOSEN,
        143 => self::DOSEN,
        144 => self::DOSEN,
        145 => self::DOSEN,
        146 => self::DIVERSE,
        147 => self::DIVERSE,
        148 => self::DIVERSE,
        149 => self::DIVERSE,
        150 => self::TEXTILIEN_ALLGEMEIN,
        151 => self::CAPS_MUETZEN,
        152 => self::DOSEN,
        153 => self::OBERBEKLEIDUNG,
        154 => self::SCHUHE,
        156 => self::DOSEN,
        157 => self::DIVERSE,
        158 => self::OBERBEKLEIDUNG,
        159 => self::OBERBEKLEIDUNG,
        160 => self::OBERBEKLEIDUNG,
        161 => self::DOSEN,
        162 => self::DOSEN,
        163 => self::DIVERSE,
        164 => self::UNTERBEKLEIDUNG,
        165 => self::TEXTILIEN_ALLGEMEIN,
        166 => self::TEXTILIEN_ALLGEMEIN,
        167 => self::DIVERSE,
        168 => self::DIVERSE,
        169 => self::DIVERSE,
        170 => self::WERKZEUG_REGENSCHIRME,
        171 => self::CAPS_MUETZEN,
        173 => self::DOSEN,
        174 => self::OBERBEKLEIDUNG,
        175 => self::UNTERBEKLEIDUNG,
        177 => self::SCHUHE,
        178 => self::DIVERSE,
        183 => self::DOSEN,
        184 => self::OBERBEKLEIDUNG,
        185 => self::OBERBEKLEIDUNG,
        186 => self::DIVERSE,
        187 => self::OBERBEKLEIDUNG,
        188 => self::OBERBEKLEIDUNG,
        189 => self::OBERBEKLEIDUNG,
        190 => self::SCHUHE,
        191 => self::DIVERSE,
        192 => self::DOSEN,
        193 => self::DOSEN,
        194 => self::TASCHEN_ALLGEMEIN,
        195 => self::TASCHEN_ALLGEMEIN,
        196 => self::TASCHEN_ALLGEMEIN,
        197 => self::TASCHEN_ALLGEMEIN,
        198 => self::TASCHEN_ALLGEMEIN,
        199 => self::TASCHEN_ALLGEMEIN,
        200 => self::TASCHEN_ALLGEMEIN,
        206 => self::RUCKSAECKE,
        207 => self::RUCKSAECKE,
        208 => self::SCHUHE,
        210 => self::TEXTILIEN_ALLGEMEIN,
        213 => self::UNTERBEKLEIDUNG,
        214 => self::OBERBEKLEIDUNG,
        215 => self::OBERBEKLEIDUNG,
        220 => self::SCHUHE,
        221 => self::OBERBEKLEIDUNG,
        227 => self::TASCHEN_ALLGEMEIN,
        228 => self::SCHUHE,
        229 => self::UNTERBEKLEIDUNG,
        235 => self::OBERBEKLEIDUNG,
        236 => self::UNTERBEKLEIDUNG,
        247 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        248 => self::PFLEGE_1_H,
        249 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        250 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        251 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        252 => self::RUCKSAECKE,
        275 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        276 => self::NAHRUNG,
        277 => self::KOCHER,
        278 => self::KOCHER,
        280 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        281 => self::RUCKSAECKE,
        288 => self::ZELT,
        289 => self::ZELT,
        290 => self::SCHLAFSAECKE,
        291 => self::ZELT,
        295 => self::UNTERBEKLEIDUNG,
        296 => self::OBERBEKLEIDUNG,
        311 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        312 => self::RUCKSAECKE,
        313 => self::NAHRUNG,
        314 => 'Wasseraufbereitung',
        332 => self::SCHUHE,
        333 => self::UNTERBEKLEIDUNG,
        336 => self::TEXTILIEN_ALLGEMEIN,
        340 => self::UNTERBEKLEIDUNG,
        357 => self::OBERBEKLEIDUNG,
        368 => self::KOCHER,
        369 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        370 => self::LEUCHTMITTEL,
        371 => self::LEUCHTMITTEL,
        386 => self::ORIENTIERUNG_GPS,
        387 => self::PFLEGE_1_H,
        388 => self::TASCHEN_ALLGEMEIN,
        389 => self::UNTERBEKLEIDUNG,
        401 => self::AKKUS_LADEGERAETE,
        402 => self::SCHUHE,
        403 => self::TASCHEN_ALLGEMEIN,
        404 => self::SCHLAFSAECKE,
        405 => self::PFLEGE_1_H,
        409 => self::SCHUHE,
        410 => self::TASCHEN_ALLGEMEIN,
        413 => self::OBERBEKLEIDUNG,
        418 => self::DIVERSE,
        419 => self::ZELT,
        424 => self::OBERBEKLEIDUNG,
        430 => self::SCHUHE,
        431 => self::TEXTILIEN_ALLGEMEIN,
        435 => self::OBERBEKLEIDUNG,
        438 => self::OBERBEKLEIDUNG,
        439 => self::DOSEN,
        441 => self::OBERBEKLEIDUNG,
        456 => self::ORIENTIERUNG_GPS,
        457 => 'Kameras',
        458 => self::AKKUS_LADEGERAETE,
        459 => self::AKKUS_LADEGERAETE,
        460 => self::RUCKSAECKE,
        461 => self::RUCKSAECKE,
        464 => self::UNTERBEKLEIDUNG,
        466 => self::SCHUHE,
        473 => self::EISENWAREN,
        474 => self::OBERBEKLEIDUNG,
        475 => self::LEUCHTMITTEL,
        476 => self::LEUCHTMITTEL,
        477 => self::RUCKSAECKE,
        483 => self::ZELT,
        484 => self::STOECKE,
        485 => self::STOECKE,
        489 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        497 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        498 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        500 => self::PFLEGE_1_H,
        509 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        510 => self::LEUCHTMITTEL,
        519 => self::EISENWAREN,
        520 => self::SCHUHE,
        521 => self::PFLEGE_1_H,
        530 => self::TEXTILIEN_ALLGEMEIN,
        531 => self::ORIENTIERUNG_GPS,
        532 => self::WERKZEUG_REGENSCHIRME,
        569 => self::ZELT,
        570 => self::AKKUS_LADEGERAETE,
        571 => self::LEUCHTMITTEL,
        572 => self::SCHLAFSAECKE,
        573 => self::ORIENTIERUNG_GPS,
        574 => self::KOCHER,
        576 => self::MATTEN_BETTEN,
        577 => self::SCHLAFSAECKE,
        578 => self::DIVERSE,
        579 => self::TEXTILIEN_ALLGEMEIN,
        580 => self::MOEBEL,
        582 => self::MOEBEL,
        583 => self::TEXTILIEN_ALLGEMEIN,
        584 => self::PFLEGE_1_H,
        585 => self::TASCHEN_ALLGEMEIN,
        586 => self::TEXTILIEN_ALLGEMEIN,
        587 => self::TEXTILIEN_ALLGEMEIN,
        588 => self::OBERBEKLEIDUNG,
        589 => self::OBERBEKLEIDUNG,
        590 => self::DIVERSE,
        591 => self::TEXTILIEN_ALLGEMEIN,
        592 => self::TEXTILIEN_ALLGEMEIN,
        593 => self::TEXTILIEN_ALLGEMEIN,
        594 => self::TEXTILIEN_ALLGEMEIN,
        596 => self::EISENWAREN,
        601 => self::ORIENTIERUNG_GPS,
        605 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        608 => self::SCHUHE,
        609 => self::RUCKSAECKE,
        615 => self::ORIENTIERUNG_GPS,
        616 => self::ORIENTIERUNG_GPS,
        617 => self::OBERBEKLEIDUNG,
        620 => self::CAPS_MUETZEN,
        624 => self::RUCKSAECKE,
        625 => self::CAPS_MUETZEN,
        631 => self::TEXTILIEN_ALLGEMEIN,
        633 => self::SCHUHE,
        634 => self::OBERBEKLEIDUNG,
        635 => self::UNTERBEKLEIDUNG,
        636 => self::OBERBEKLEIDUNG,
        640 => self::UNTERBEKLEIDUNG,
        643 => self::OBERBEKLEIDUNG,
        644 => self::OBERBEKLEIDUNG,
        647 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        649 => self::UNTERBEKLEIDUNG,
        651 => self::TEXTILIEN_ALLGEMEIN,
        655 => self::DIVERSE,
        656 => self::TEXTILIEN_ALLGEMEIN,
        663 => self::SCHUHE,
        664 => self::SCHUHE,
        665 => self::TEXTILIEN_ALLGEMEIN,
        666 => self::SCHUHE,
        669 => self::MATTEN_BETTEN,
        671 => self::UNTERBEKLEIDUNG,
        677 => self::UNTERBEKLEIDUNG,
        683 => self::SCHUHE,
        686 => self::ZELT,
        689 => self::SCHLAFSAECKE,
        697 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        698 => self::MATTEN_BETTEN,
        701 => self::MATTEN_BETTEN,
        706 => self::MOEBEL,
        707 => self::MOEBEL,
        708 => self::ZELT,
        709 => self::DIVERSE,
        714 => self::OBERBEKLEIDUNG,
        719 => self::DIVERSE,
        721 => self::DIVERSE,
        725 => self::OBERBEKLEIDUNG,
        729 => self::UNTERBEKLEIDUNG,
        733 => self::UNTERBEKLEIDUNG,
        740 => self::MATTEN_BETTEN,
        741 => self::SCHUHE,
        751 => self::OBERBEKLEIDUNG,
        754 => self::MATTEN_BETTEN,
        755 => self::OBERBEKLEIDUNG,
        758 => self::OBERBEKLEIDUNG,
        764 => self::UNTERBEKLEIDUNG,
        767 => self::UNTERBEKLEIDUNG,
        768 => self::TEXTILIEN_ALLGEMEIN,
        786 => self::ORIENTIERUNG_GPS,
        787 => self::ORIENTIERUNG_GPS,
        788 => self::ORIENTIERUNG_GPS,
        789 => self::UNTERBEKLEIDUNG,
        790 => self::OBERBEKLEIDUNG,
        794 => self::TEXTILIEN_ALLGEMEIN,
        797 => self::DIVERSE,
        799 => self::UNTERBEKLEIDUNG,
        801 => self::SCHUHE,
        803 => self::ZELT,
        806 => self::DIVERSE,
        815 => self::TASCHEN_ALLGEMEIN,
        816 => self::UNTERBEKLEIDUNG,
        828 => self::ZELT,
        836 => self::SCHUHE,
        838 => self::TASCHEN_ALLGEMEIN,
        840 => self::ZELT,
        841 => self::OBERBEKLEIDUNG,
        842 => self::OBERBEKLEIDUNG,
        846 => self::OBERBEKLEIDUNG,
        851 => self::KOCHER,
        852 => self::KOCHER,
        856 => self::PFLEGE_1_H,
        857 => self::DOSEN,
        860 => self::DIVERSE,
        864 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        878 => self::DOSEN,
        882 => self::UNTERBEKLEIDUNG,
        886 => self::LEUCHTMITTEL,
        888 => self::KOCHER,
        892 => self::KOCHER,
        894 => self::TEXTILIEN_ALLGEMEIN,
        900 => self::STOECKE,
        910 => self::DIVERSE,
        912 => self::DIVERSE,
        914 => self::STOECKE,
        920 => self::RUCKSAECKE,
        924 => self::OBERBEKLEIDUNG,
        934 => self::STOECKE,
        936 => self::STOECKE,
        938 => self::STOECKE,
        940 => self::SCHUHE,
        942 => self::TEXTILIEN_ALLGEMEIN,
        944 => self::TASCHEN_ALLGEMEIN,
        946 => self::NAHRUNG,
        948 => self::DIVERSE,
        950 => self::DIVERSE,
        952 => self::DIVERSE,
        954 => self::NAHRUNG,
        956 => self::TEXTILIEN_ALLGEMEIN,
        958 => self::DIVERSE,
        960 => self::DIVERSE,
        962 => self::STOECKE,
        964 => self::STOECKE,
        966 => self::TASCHEN_ALLGEMEIN,
        968 => self::SCHUHE,
        970 => self::DIVERSE,
        980 => self::ZELT,
        988 => 'kartuschen-flaschen',
        990 => self::TASCHEN_ALLGEMEIN,
        993 => self::BUECHER_MEDIEN,
        997 => self::ACCESSOIRES,
        999 => self::ACCESSOIRES,
        1001 => self::BUECHER_MEDIEN,
        1003 => self::BUECHER_MEDIEN,

        52 => self::BUECHER_MEDIEN,
        57 => self::MARKER_REFILL,
        60 => self::BUECHER_MEDIEN,
        61 => self::BUECHER_MEDIEN,
        62 => self::BUECHER_MEDIEN,
        63 => self::BUECHER_MEDIEN,
        64 => self::BUECHER_MEDIEN,
        65 => self::BUECHER_MEDIEN,
        66 => self::BUECHER_MEDIEN,
        67 => self::BUECHER_MEDIEN,
        68 => self::BUECHER_MEDIEN,
        70 => self::BUECHER_MEDIEN,
        71 => self::BUECHER_MEDIEN,
        72 => self::BUECHER_MEDIEN,
        99 => self::MARKER_REFILL,
        105 => self::BUECHER_MEDIEN,
        132 => self::ACCESSOIRES,
        137 => self::MARKER_REFILL,
        138 => self::MARKER_REFILL,
        139 => self::MARKER_REFILL,
        140 => self::MARKER_REFILL,
        141 => self::MARKER_REFILL,
        142 => self::MARKER_REFILL,
        143 => self::MARKER_REFILL,
        144 => self::MARKER_REFILL,
        145 => self::MARKER_REFILL,
        146 => self::ACCESSOIRES,
        147 => self::BUECHER_MEDIEN,
        148 => self::BUECHER_MEDIEN,
        149 => self::BUECHER_MEDIEN,
        152 => self::MARKER_REFILL,
        162 => self::MARKER_REFILL,
        168 => self::ACCESSOIRES,
        169 => self::ACCESSOIRES,
        178 => self::BUECHER_MEDIEN,
        186 => self::BUECHER_MEDIEN,
        191 => self::BUECHER_MEDIEN,
        193 => self::MARKER_REFILL,
        418 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        655 => self::ACCESSOIRES,
        857 => self::MARKER_REFILL,

        1005 => self::SCHLAFSAECKE,
        1007 => self::WERKZEUG_REGENSCHIRME,
        1009 => self::SCHLAFSAECKE,
        1011 => self::SCHLAFSAECKE,
        1013 => self::SCHLAFSAECKE,
        1015 => self::SCHUHE,
        1017 => self::TEXTILIEN_ALLGEMEIN,
        1019 => self::LEUCHTMITTEL,
        1021 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        1023 => self::STOECKE,
        1025 => self::SCHUHE,
        1027 => self::RUCKSAECKE,
        1029 => self::RUCKSAECKE,
        1031 => self::SCHUHE,
        1033 => self::SCHUHE,
        1035 => self::SCHUHE,
        1037 => self::SCHUHE,
        1039 => self::SCHUHE,
        1041 => self::SCHUHE,

        1049 => self::SCHUHE,
        1071 => 'Wasserbehälter-Küchenutensilien',
        1081 => self::SCHLAFSAECKE,
        1085 => self::SCHLAFSAECKE,
        1089 => self::RUCKSAECKE,
        1091 => self::SCHLAFSAECKE,
        1095 => 'Wasserbehälter-Küchenutensilien',

        1152 => self::ACCESSOIRES,
        1150 => self::TASCHEN_ALLGEMEIN,
        1154 => self::SCHUHE,
        1220 => self::UNTERBEKLEIDUNG,
        1146 => self::SCHUHE,
        1115 => self::SCHUHE,

        1495 => self::SCHUHE,
        1093 => self::SCHUHE,
        1117 => self::SCHUHE,
        1099 => self::SCHUHE,
        1097 => self::SCHUHE,
        1521 => self::SCHUHE,
        1624 => self::SCHUHE,
        1605 => self::TASCHEN_ALLGEMEIN,
        1613 => self::OBERBEKLEIDUNG,
        1611 => self::TEXTILIEN_ALLGEMEIN,
        1630 => self::MATTEN_BETTEN,
        1632 => self::MATTEN_BETTEN,
        1636 => self::MATTEN_BETTEN,

        1868 => self::OBERBEKLEIDUNG,
        1087 => self::TEXTILIEN_ALLGEMEIN,
        1128 => self::TEXTILIEN_ALLGEMEIN,
        1523 => self::TEXTILIEN_ALLGEMEIN,
        1901 => self::UNTERBEKLEIDUNG,
        1769 => self::OBERBEKLEIDUNG,
        1931 => self::OBERBEKLEIDUNG,
        1928 => self::OBERBEKLEIDUNG,
        1772 => self::SCHUHE,
        1691 => self::SCHUHE,
        1885 => self::SCHUHE,
        1700 => self::SCHUHE,
        1657 => self::WASSERBEHAELTER_KUECHENUTENSILIEN,
        1130 => self::MARKER_REFILL,
        1916 => self::OBERBEKLEIDUNG,
        1919 => self::OBERBEKLEIDUNG,
        1922 => self::OBERBEKLEIDUNG,
        1929 => self::UNTERBEKLEIDUNG,
        1501 => self::OBERBEKLEIDUNG,
        1961 => self::OBERBEKLEIDUNG,
        1964 => self::UNTERBEKLEIDUNG,
        1949 => self::UNTERBEKLEIDUNG,
        1970 => self::OBERBEKLEIDUNG,
        1973 => self::UNTERBEKLEIDUNG,
    ];

    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {

        $this->apiM = $apiM;
    }

    /**
     * @param $path string
     *
     * @return \Pimcore\Model\Object\AbstractObject
     * @throws \Exception
     */
    public function getFolderArtikelFamilie($path)
    {

        return $this->getFolder($path, static::ARTIKELFAMILIEN);
    }

    /**
     * @param $path string
     *
     * @return \Pimcore\Model\Object\AbstractObject
     * @throws \Exception
     */
    public function getFolderProduktGruppe($path)
    {

        return $this->getFolder($path, static::PRODUKTGRUPPEN);
    }

    /**
     * @param $path
     * @param $type
     *
     * @return \Pimcore\Model\Object\AbstractObject
     * @throws \Exception
     */
    private function getFolder($path, $type)
    {

        /**
         * mapping
         */
        $folder = Folder::getByPath("/artikel/$path/$type/");

        if (empty($folder)) {
            throw new \Exception("path '/artikel/$path/$type/' not found");
        }

        return $folder;
    }

    /**
     * @param $parent
     * @param $path
     *
     * @return null|Folder
     */
    private function createDir($parent, $path)
    {

        if (empty(Folder::getByPath("/$parent/$path/"))) {
            return Folder::create([
                "o_parentId" => Folder::getByPath("/$parent/")->getId(),
                "o_creationDate" => time(),
                "o_key" => $path,
            ]);
        }

        return null;
    }

    /**
     * @return string
     */
    private function generateDefaultDirectories()
    {

        // Create default directories if they do not exist
        $this->createDir('artikel', $this->mappingShopDefault);
        $this->createDir('artikel/' . $this->mappingShopDefault, static::PRODUKTGRUPPEN);
        $this->createDir('artikel/' . $this->mappingShopDefault, static::ARTIKELFAMILIEN);

        return $this->mappingShopDefault;
    }

    /**
     * @param $categoryId
     *
     * @return mixed|string
     */
    public function categoryMappingShop($categoryId)
    {

        if (empty($this->mappingShop[$categoryId])) {
            return $this->generateDefaultDirectories();
        }

        /**
         * mapping is valid but directory does not exist!
         */
        if (empty(Folder::getByPath("/artikel/" . $this->mappingShop[$categoryId] . "/"))) {
            return $this->generateDefaultDirectories();
        }

        /**
         * mapping is correct and directory exists
         */
        return $this->mappingShop[$categoryId];
    }

    /**
     * @param string $articleFamilyCode
     * @param string $path
     * @return null|ArtikelFamilie|ProduktGruppe
     * @throws \Exception
     */
    public function loadArtikelFamilie($articleFamilyCode, $path)
    {
        return $this->loadObject($articleFamilyCode, null, $path, static::ARTIKELFAMILIEN);
    }

    /**
     * @param string $pgc
     * @param string $path
     * @return null|ArtikelFamilie|ProduktGruppe
     * @throws \Exception
     */
    public function loadProduktGruppe($pgc, $path)
    {
        return $this->loadObject(null, $pgc, $path, static::PRODUKTGRUPPEN);
    }

    /**
     * Return new or existing object
     *
     * @param string $articleFamilyCode
     * @param string $pgc
     * @param string $path
     * @param string $type
     *
     * @return null|ArtikelFamilie|Produktgruppe
     * @throws \Exception
     */
    public function loadObject($articleFamilyCode, $pgc, $path, $type)
    {

        $folder = '';
        $objectListing = null;

        if ($type == static::ARTIKELFAMILIEN) {
            $objectListing = new ArtikelFamilie\Listing();
            $objectListing->setCondition('o_key = ?', $articleFamilyCode);
            $folder = $this->getFolderArtikelFamilie($path);
        } elseif ($type == static::PRODUKTGRUPPEN) {
            $objectListing = new ProduktGruppe\Listing();
            $objectListing->setCondition('o_key = ?', $pgc);
            $folder = $this->getFolderProduktGruppe($path);
        }

        if ($objectListing === null) {
            return null;
        }

        $objectListing->setUnpublished(true);
        $objectListing->load();
        $object = null;

        /**
         * Object already exists in PimCore
         */
        if ($objectListing->count() === 1) {
            $object = $objectListing->current();
        }

        if ($object === null) {
            /**
             * Object does not exists in PimCore
             */
            if ($type == static::ARTIKELFAMILIEN) {
                $object = new ArtikelFamilie();
                $object->setKey(File::getValidFilename($articleFamilyCode));
            } elseif ($type == static::PRODUKTGRUPPEN) {
                $object = new ProduktGruppe();
                $object->setKey(File::getValidFilename($pgc));
            }
        }

        $object->setParentId($folder->getId());

        return $object;
    }

    /**
     * @param $articleFamilyCode
     *
     * @return null|Link
     */
    public function buildLinkArtikelFamilieBigTree($articleFamilyCode)
    {

        return $this->buildLink($articleFamilyCode, static::ARTIKELFAMILIEN, 101);
    }

    /**
     * @param $familyId int
     *
     * @return null|Link
     */
    public function buildLinkProduktGruppeBigTree($familyId)
    {

        return $this->buildLink($familyId, static::PRODUKTGRUPPEN, 101);
    }

    /**
     * @param $articleFamilyCode
     *
     * @return null|Link
     */
    public function buildLinkArtikelFamilieStylefile($articleFamilyCode)
    {

        return $this->buildLink($articleFamilyCode, static::ARTIKELFAMILIEN, 1);
    }

    /**
     * @param $familyId int
     *
     * @return null|Link
     */
    public function buildLinkProduktGruppeStylefile($familyId)
    {

        return $this->buildLink($familyId, static::PRODUKTGRUPPEN, 1);
    }

    /**
     * @param $articleId        string
     * @param $type             string
     * @param $contentId        int
     *
     * @return Link|null
     */
    public function buildLink($articleId, $type, $contentId)
    {

        if (!in_array($type, [static::ARTIKELFAMILIEN, static::PRODUKTGRUPPEN])) {
            return null;
        }

        if (!in_array($contentId, [1, 101])) {
            return null;
        }

        $link = '';

        if ($contentId == 1) {
            $link .= '<a href="http://www.stylefile.de/';
        } elseif ($contentId == 101) {
            $link .= '<a href="http://www.bigtree.de/';
        }

        if ($type == static::ARTIKELFAMILIEN) {
            if ($contentId == 1) {
                $link .= 'streetwear-sneaker-graffiti/?keywords=' . $articleId . '" target="_blank">zum Shop</a>';
            } elseif ($contentId == 101) {
                $link .= 'abenteuer-und-outdoor-ausruestung/?keywords=' . $articleId . '" target="_blank">zum Shop</a>';
            }
        } elseif ($type == static::PRODUKTGRUPPEN) {
            $link .= 'xxx-fid-' . $articleId . '.html" target="_blank">zum Shop</a>';
        }

        $pimCoreLink = new Link();

        return $pimCoreLink->setText($link);
    }

    /**
     * @param ProduktGruppe $productGroup
     * @param ShopData      $shopData
     * @param Availability  $availability
     * @return ProduktGruppe
     * @throws \Zend_Date_Exception
     */
    public function setDataProduktGruppe(ProduktGruppe $productGroup, ShopData $shopData, Availability $availability)
    {
        // Process ProductGroup data
        $productGroup->setDisplayArtikelname($shopData->getArticleName());
        $productGroup->setDisplayFarbe($shopData->getColor());
        $productGroup->setDisplayPreis($shopData->getPrice());
        $productGroup->setDisplayPreisUvp($shopData->getPriceUvp());
        $productGroup->setDisplayLagerbestand($availability->getRealQty());
        $productGroup->setDisplayQuantityOnStock($availability->getQuantityOnStock());

        // if date is 1970-01-01 01:00:00 no data will be shown in form
        $activationDate = new Date($shopData->getActivationDate());
        $productGroup->setDisplayAktivierungsdatum($activationDate);
        $productGroup->setDisplayArtikelID($availability->getPgc());

        $productGroup->setDisplayComingSoon(0);
        if ($availability->getShippingState() == 21 || $availability->getShippingState() == 22) {
            $productGroup->setDisplayComingSoon(1);
        }
        // set shopGroup information (STYLEF, BIGT or MIXED)
        if (null !== $availability->getItem() && null !== $availability->getItem()->getShopGroup()) {
            $productGroup->setDisplayShopGruppe($availability->getItem()->getShopGroup());
        }
        $productGroup->setDisplayShippingState($availability->getShippingState());
        $productGroup->setDisplayTextilkennzeichnung(str_replace('<br>', "\n", $shopData->getCareLabeling()));
        $productGroup->setDisplayShopID(str_replace('<br>', "\n", $shopData->getShopAssignment()));
        $productGroup->setDisplayKreditorenartikelnummer($shopData->getVendorItemNo());
        $productGroup->setDisplayVollAbgleiche(str_replace('<br>', "\n", $shopData->getFullUpdate()));
        $productGroup->setDisplayTeilAbgleiche(str_replace('<br>', "\n", $shopData->getPartialUpdate()));
        $productGroup->setDisplayLinkBigtree($this->buildLinkProduktGruppeBigTree($shopData->getFamilyLinkId()));
        $productGroup->setDisplayLinkStylefile($this->buildLinkProduktGruppeStylefile($shopData->getFamilyLinkId()));

        return $productGroup;
    }

    /**
     * @param ArtikelFamilie $articleFamily
     * @param ProduktGruppe  $productGroup
     * @param ShopData       $shopData
     * @param Availability   $availability
     *
     * @return ArtikelFamilie
     */
    public function setDataArtikelFamilie(
        ArtikelFamilie $articleFamily,
        ProduktGruppe $productGroup,
        ShopData $shopData,
        Availability $availability
    ) {

        $articleFamily->setDisplayArtikelname($shopData->getArticleName());
        $articleFamily->setDisplayCode($shopData->getArticleFamilyCode());
        $articleFamily->setDisplayVollAbgleiche(str_replace('<br>', "\n", $shopData->getFullUpdate()));
        $articleFamily->setDisplayTeilAbgleiche(str_replace('<br>', "\n", $shopData->getPartialUpdate()));
        $articleFamily->setDisplayKategorie($shopData->getCategoryName());
        $articleFamily->setDisplayTextilkennzeichnung(str_replace('<br>', "\n", $shopData->getCareLabeling()));
        $articleFamily->setDisplayLinkBigtree($this->buildLinkArtikelFamilieBigTree($shopData->getArticleFamilyCode()));
        $articleFamily->setDisplayLinkStylefile($this->buildLinkArtikelFamilieStylefile($shopData->getArticleFamilyCode()));

        // STYLEFILE-2631: ArtikelFamilie -> Übersicht zugehöriger Produktgruppen in extra Tab
        $produktGruppen = $articleFamily->getDisplayProduktgruppen();

        $addProduktGruppe = true;
        /**
         * @var ProduktGruppe $artikelFamilieProduktGruppe
         */
        foreach ($produktGruppen as $produktGruppe) {
            if ($produktGruppe->getDisplayArtikelID() == $availability->getPgc()) {
                $addProduktGruppe = false;
                break;
            }
        }
        if ($addProduktGruppe) {
            /**
             * Produktgruppe dem Array der ArtikelFamilieProduktGruppen hinzufügen
             */
            $produktGruppen[] = $productGroup;
            $articleFamily->setDisplayProduktgruppen($produktGruppen);
        }

        return $articleFamily;
    }

    /**
     * @param $xmlString
     *
     * @return \ImportArticle\Model\Request\Item
     * @throws \Exception
     */
    public function deserializeXML($xmlString)
    {

        return $this->apiM->serializer()->deserialize($xmlString, 'ImportArticle\Model\Request\Item', 'xml');
    }

    /**
     * @param $xmlString
     *
     * @return bool
     * @throws \Exception
     */
    public function run($xmlString)
    {
        $itemTransferObjects = $this->deserializeXML($xmlString);
        $timeTotalStart = microtime(true);
        $deserializeData = $itemTransferObjects->getElements();

        $status = [];
        foreach ($deserializeData as $itemTransferObject) {
            try {
                $timeStart = microtime(true);

                $shopData = $itemTransferObject->getShopData();
                $availability = $itemTransferObject->getAvailability();

                $path = $this->categoryMappingShop($shopData->getCategoryId());

                // Load existing or new objects
                $articleFamily = $this->loadArtikelFamilie($shopData->getArticleFamilyCode(), $path);
                $productGroup = $this->loadProduktGruppe($availability->getPgc(), $path);

                // Process ArticleFamily data
                $this->apiM->logger()->debug(sprintf('Try to process ArtikelFamilie/Produktgruppe: %s!',
                    $shopData->getArticleFamilyCode())); // \Zend_Log::DEBUG

                $this->setDataProduktGruppe($productGroup, $shopData, $availability)->save();
                $this->setDataArtikelFamilie($articleFamily, $productGroup, $shopData, $availability)->save();

                $timeEnd = microtime(true);
                $executionTime = ($timeEnd - $timeStart);

                $this->apiM->logger()->debug(sprintf('Current payload: %s', print_r($itemTransferObject, true)));
                $this->apiM->logger()->debug(sprintf('ArtikelFamilie/Produktgruppe: %s saved! Processing time: %f s',
                    $shopData->getArticleFamilyCode(), $executionTime));

                $status[$availability->getPgc()] = ['success' => true, 'message' => ''];
            } catch (\Exception $e) {
                $return[$availability->getPgc()] = ['success' => false, 'message' => $e->getMessage()];

                return $this->apiM->serializer()->serialize($return, 'json');
            }
        }
        $timeTotalEnd = microtime(true);
        $executionTotalTime = ($timeTotalEnd - $timeTotalStart);
        $this->apiM->logger()->debug(sprintf('Total processing time: %f s', $executionTotalTime));

        return $this->apiM->serializer()->serialize($status, 'json');
    }
}
