<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Publikat\Pimcore\SaveChangesTrackingBundle\Worker;

use Publikat\Pimcore\SaveChangesTrackingBundle\Api\Manager as ApiManager;
use Carbon\Carbon;
use Pimcore\Model\Element\ElementInterface;
use Pimcore\Model\Object\ArtikelFamilie;
use Pimcore\Model\Object\ChangesTracking as ChangesTrackingModel;
use Pimcore\Model\Object\Folder;
use Pimcore\Model\Object\Localizedfield;
use Pimcore\Model\Object\Produktgruppe;

/**
 * Class ChangesTracking
 *
 * @package Publikat\Pimcore\SaveChangesTrackingBundle\Worker
 */
class ChangesTracking
{
    const STYLEFILE = 'stylefile';
    const BIGTREE = 'bigtree';

    /**
     * @var ApiManager
     */
    protected $apiManager;

    /**
     * @param ApiManager $apiManager
     */
    public function __construct(ApiManager $apiManager)
    {
        $this->apiManager = $apiManager;
    }

    /**
     * handle
     *
     * @param ElementInterface $object
     *
     * @return ElementInterface
     * @throws \Pimcore\Model\Element\ValidationException
     */
    public function run($object)
    {
        if ($object instanceof ProduktGruppe || $object instanceof ArtikelFamilie) {
            $this->saveChanges($object, $this->getOldData($object));
        }

        return $object;
    }

    /**
     * @param ElementInterface $newObject
     *
     * @return array|null
     */
    public function getOldData($newObject)
    {
        $pimcoreObjectIndex = $newObject->getClassId();
        $databaseConnection = $this->apiManager->database();
        $query = "SELECT object_localized_data_$pimcoreObjectIndex.* FROM object_localized_data_$pimcoreObjectIndex
                  LEFT JOIN object_$pimcoreObjectIndex ON
                  (object_$pimcoreObjectIndex.oo_id = object_localized_data_$pimcoreObjectIndex.ooo_id)
                  WHERE object_$pimcoreObjectIndex.oo_id =" . $newObject->getId();
        $result = $databaseConnection->query($query)->fetchAll();
        if (count($result) > 0) {
            return $result;
        }

        return null;
    }

    /**
     * @param Folder $folder
     *
     * @return $this
     * @throws \Exception
     * @throws \Pimcore\Model\Element\ValidationException
     */
    public function createFolder(Folder $folder)
    {
        $rootFolder = Folder::getByPath("/");

        return $folder
            ->setParentId($rootFolder->getId())
            ->setKey('changes-tracking')
            ->save();
    }

    /**
     * @return ChangesTracking
     * @throws \Pimcore\Model\Element\ValidationException
     */
    public function getPath()
    {
        $changesTrackingFolder = Folder::getByPath("/changes-tracking/");
        if (is_null($changesTrackingFolder)) {
            $newFolder = $this->apiManager->folder();
            $changesTrackingFolder = $this->createFolder($newFolder);
        }

        return $changesTrackingFolder;
    }

    /**
     * @param $newObject ElementInterface
     * @param $oldData   []
     *
     * @return array
     */
    public function getChanges($newObject, $oldData)
    {
        $changes = [];
        $changes[static::STYLEFILE] = false;
        $changes[static::BIGTREE] = false;
        foreach ($oldData as $oldLanguageData) {
            $language = $oldLanguageData['language'];
            $changes[$language] = 0;
            // load saved object data
            $localizedFields = $newObject->getLocalizedfields()->getInternalData()[$language];
            $keys = array_keys($localizedFields);
            if (!isset($keys)) {
                continue;
            }

            foreach ($keys as $analyzeItem) {
                // STYLEFILE-5458 : Ignore arrays like on bullet points (blocks)
                if (is_array($localizedFields[$analyzeItem])) {
                    continue;
                }
                // do strings have equal length?
                $lengthOld = strlen($oldLanguageData[$analyzeItem]);
                $lengthNew = strlen($localizedFields[$analyzeItem]);
                if ($lengthOld == $lengthNew) {
                    // calculate hamming distance
                    if ($oldLanguageData[$analyzeItem] !== $localizedFields[$analyzeItem]) {
                        $valueChange = $this->calculateDistance(
                            $oldLanguageData[$analyzeItem], $localizedFields[$analyzeItem]
                        );
                        $changes[$language] += $valueChange;
                        $changes = $this->checkShopGroupChanges($analyzeItem, $changes, $valueChange);
                    }
                } else {
                    $valueChange = ($lengthNew - $lengthOld);
                    // just track difference in length
                    $changes[$language] += $valueChange;
                    $changes = $this->checkShopGroupChanges($analyzeItem, $changes, $valueChange);
                }
            }
        }

        return $changes;
    }

    /**
     * @param $analyzeItem string
     * @param $changes     []
     * @param $valueChange int
     *
     * @return mixed
     */
    public function checkShopGroupChanges($analyzeItem, $changes, $valueChange)
    {
        if ((strpos(strtolower($analyzeItem), static::STYLEFILE) != null) && ($valueChange > 0 || $valueChange < 0)) {
            $changes[static::STYLEFILE] = true;
        }
        if ((strpos(strtolower($analyzeItem), static::BIGTREE) != null) && ($valueChange > 0 || $valueChange < 0)) {
            $changes[static::BIGTREE] = true;
        }

        return $changes;
    }

    /**
     * @param $stateOld
     * @param $stateNew
     *
     * @return int
     */
    public function calculateDistance($stateOld, $stateNew)
    {
        $a1 = str_split($stateOld);
        $a2 = str_split($stateNew);
        $dh = 0;
        for ($i = 0; $i < count($a1); $i++) {
            if ($a1[$i] != $a2[$i]) {
                $dh++;
            }
        }

        return $dh;
    }

    /**
     * @param $object
     * @param $oldData
     *
     * @throws \Pimcore\Model\Element\ValidationException
     */
    public function saveChanges($object, $oldData)
    {
        $changesTrackingFolder = $this->getPath();

        $changes = $this->getChanges($object, $oldData);

        if (!is_null($object->getUserModification()) && $object->getUserModification() != '' && is_array($changes) && array_sum($changes) > 0) {
            $changesTracking = (new ChangesTrackingModel())
                ->setUser($object->getUserModification())
                ->setDateModified(Carbon::createFromTimestamp(time()))
                ->setDateModifiedYear(date("Y"))
                ->setDateModifiedWeek(date("W"))
                ->setObjectId($object->getId())
                ->setKey(time() . $object->getUserModification())
                ->setParentId($changesTrackingFolder->getId())
                ->setAmountChangesTotal(array_sum($changes))
                ->setIsBigtreeChange($changes[static::BIGTREE])
                ->setIsStylefileChange($changes[static::STYLEFILE]);

            $localizedField = new Localizedfield();
            $amountChangesArray = [];
            // process every original language
            foreach ($object->getLocalizedfields()->getInternalData() as $languageToken => $languageContent) {
                $amountChangesArray[$languageToken] = ['amountChanges' => $changes[$languageToken]];
            }

            // only save real changes (negative and positive)
            if (array_sum($changes) != 0) {
                $changesTracking->save();

                $localizedField->setItems($amountChangesArray);
                $localizedField->setObject($changesTracking);
                $localizedField->save();

                $changesTracking->setLocalizedfields($localizedField);
                $changesTracking->save();
            }
        }
    }
}
