<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Asset\RabbitMq\Handler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Psr\Log\LoggerInterface;

/**
 * Class LocalFileHandler
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Asset\RabbitMq\Handler
 */
class LocalFileHandler implements HandlerInterface
{
    /**
     * Source domain where to pull the assets
     * e. g. http://pimcore.local (path is added depending on asset)
     *
     * @var string
     */
    private $sourceUrl;

    /**
     * Web root directory of target system
     * e. g. /var/www/mydomain/htdocs
     *
     * @var string
     */
    private $targetFolder;

    /**
     * Url path to target system
     * e. g. /assets
     *
     * @var string
     */
    private $targetPath;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * LocalFileHandler constructor.
     * @param string $sourceUrl
     * @param string $targetFolder
     * @param string $targetPath
     * @param LoggerInterface $logger
     */
    public function __construct(string $sourceUrl, string $targetFolder, string $targetPath, LoggerInterface $logger)
    {
        $this->sourceUrl = $sourceUrl;
        $this->targetFolder = $targetFolder;
        $this->targetPath = $targetPath;
        $this->logger = $logger;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector): bool
    {
        return true;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function publish(DataCollectorInterface $collector): bool
    {
        $assetSourceUrl = $this->getSourceUrl($collector);
        $source = null;
        try {
            $source = fopen($assetSourceUrl, 'r');
        } catch (\Exception $exception) {
            $this->logger->warning(sprintf(
                'Cannot open asset source path "%s" for reading.',
                $assetSourceUrl
            ));
        }

        $targetFile = $this->getTargetFilename($collector);
        $targetFullpath = $this->targetFolder . $this->targetPath . $targetFile;
        if (!file_exists(dirname($targetFullpath))) {
            try {
                mkdir(dirname($targetFullpath), 0777, true);
            } catch (\Exception $e) {
                $this->logger->warning(sprintf(
                    'Cannot open create target folder "%s".',
                    dirname($targetFullpath)
                ));
            }
        }

        $target = null;
        try {
            $target = fopen($targetFullpath, 'w');
        } catch (\Exception $exception) {
            $this->logger->warning(sprintf(
                'Cannot open asset target path "%s" for writing.',
                $targetFullpath
            ));
        }

        if (!is_resource($source) || !is_resource($target)) {
            return false;
        }

        stream_copy_to_stream($source, $target);
        fclose($source);
        fclose($target);

        $this->updateHtAccess($collector->getItem('id')->getValue(), $targetFile);
        $this->updateManifestJson($collector->getItem('id')->getValue(), $collector->getItem('path')->getValue() . $collector->getItem('filename')->getValue());

        return true;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function delete(DataCollectorInterface $collector): bool
    {
        $targetFile = $this->getTargetFilename($collector);
        $targetFullpath = $this->targetFolder . $this->targetPath . $targetFile;
        if (file_exists($targetFullpath)) {
            unlink($targetFullpath);

            $path = $targetFullpath;
            while (($path = dirname($path)) && $path !== $this->targetFolder . $this->targetPath) {
                if (!(new \FilesystemIterator($path))->valid()) {
                    rmdir($path);
                }
            }
        }

        $this->updateHtAccess($collector->getItem('id')->getValue(), '');
        $this->updateManifestJson($collector->getItem('id')->getValue(), '');

        return true;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return string
     */
    private function getSourceUrl(DataCollectorInterface $collector): string
    {
        $path = $collector->getItem('path')->getValue();
        $filename = $collector->getItem('filename')->getValue();

        return $this->sourceUrl .
            implode('/', array_map('rawurlencode', explode('/', $path))) .
            rawurlencode($filename);
    }

    /**
     * @param DataCollectorInterface $collector
     * @return string
     */
    private function getTargetFilename(DataCollectorInterface $collector): string
    {
        $path = preg_replace('~(.)~', '/$1', sprintf('%010s', $collector->getItem('id')->getValue()));
        $extension = preg_replace('~(.+)?\.(\w+)(\?.+)?~', '$2', $collector->getItem('filename')->getValue());

        return $path . '.' . $extension;
    }

    /**
     * @param int $assetId
     * @param string $targetFile
     */
    private function updateHtAccess($assetId, $targetFile)
    {

        $rewritePrefix = 'RewriteRule ^/?' . $assetId . '- ';

        $htAccess = $this->targetFolder . $this->targetPath . '/.htaccess';
        if (file_exists($htAccess)) {
            $lines = file($htAccess, FILE_IGNORE_NEW_LINES);
            $lines = array_filter($lines, function($line) use ($rewritePrefix) {
                return 0 !== strpos($line, $rewritePrefix);
            });
        } else {
            $lines = [
                'RewriteEngine On',
                'RewriteBase ' . $this->targetPath,
            ];
        }

        if (0 < strlen($targetFile)) {
            $lines[] = $rewritePrefix . $this->targetPath . $targetFile;
        }

        file_put_contents($htAccess, implode(PHP_EOL, $lines));
    }

    /**
     * @param int $assetId
     * @param string $targetPath
     */
    private function updateManifestJson($assetId, $targetPath)
    {
        $manifestJson = $this->targetFolder . $this->targetPath . '/manifest.json';
        if (file_exists($manifestJson)) {
            $lines = json_decode(file_get_contents($manifestJson));
        } else {
            $lines = new \stdClass();
        }

        if (0 < strlen($targetPath)) {
            $lines->$assetId = $this->targetPath . '/' . $assetId . '-' . ltrim($targetPath, '/');
        } else {
            unset($lines->$assetId);
        }

        file_put_contents($manifestJson, json_encode($lines));
    }
}
