<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Asset\RabbitMq\Distributor;

use Lifestyle\Asset\RabbitMq\Handler\HandlerInterface as AssetHandlerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Lifestyle\RabbitMq\Producer\Producer\ProducerInterface;

/**
 * Class UnpublishDistributor
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Asset\RabbitMq\Distributor
 */
class UnpublishDistributor implements HandlerInterface
{
    /**
     * @var array
     */
    private $responsibleEvents;

    /**
     * @var AssetHandlerInterface
     */
    private $assetHandler;

    /**
     * @var ProducerInterface|null
     */
    private $producer;

    /**
     * UnpublishDistributor constructor.
     * @param array $responsibleEvents
     * @param AssetHandlerInterface $assetHandler
     * @param ProducerInterface|null $producer
     */
    public function __construct(
        array $responsibleEvents,
        AssetHandlerInterface $assetHandler,
        ?ProducerInterface $producer
    ) {
        $this->responsibleEvents = $responsibleEvents;
        $this->assetHandler = $assetHandler;
        $this->producer = $producer;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (
            $this->isResponsible($dataCollector) &&
            $this->assetHandler->delete($dataCollector)
        ) {
            if (null !== $this->producer) {
                $this->producer->send($dataCollector->getItemValue('event'), $dataCollector);
            }
            return true;
        }

        return false;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return
            null !== ($eventName = $dataCollector->getItemValue('event')) &&
            in_array(strtolower($eventName), $this->responsibleEvents) &&
            $this->assetHandler->isResponsible($dataCollector);
    }
}
