<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Asset\RabbitMq\Handler;

use Lifestyle\Asset\RabbitMq\Exception\ExceptionInterface;
use Lifestyle\Asset\RabbitMq\Exception\FilesystemException;
use Lifestyle\DataCollector\DataCollectorInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Filesystem\Exception\IOExceptionInterface;
use Symfony\Component\Filesystem\Filesystem;

/**
 * Class DefaultFileHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Asset\RabbitMq\Handler
 */
class DefaultFileHandler implements HandlerInterface
{
    /**
     * Source domain where to pull the assets
     * e. g. http://pimcore.local (path is added depending on asset)
     *
     * @var string
     */
    private $sourceUrl;

    /**
     * Web root directory of target system
     * e. g. /var/www/mydomain/htdocs
     *
     * @var string
     */
    private $targetFolder;

    /**
     * Url path to target system
     * e. g. /assets
     *
     * @var string
     */
    private $targetPath;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var array
     */
    private $streamContext = [
        'ssl' => [
            'verify_peer' => true,
        ]
    ];

    /**
     * DefaultFileHandler constructor.
     *
     * @param string $sourceUrl
     * @param string $targetFolder
     * @param string $targetPath
     * @param LoggerInterface $logger
     * @param string $caCert
     */
    public function __construct(string $sourceUrl, string $targetFolder, string $targetPath, LoggerInterface $logger, string $caCert)
    {
        $this->sourceUrl = $sourceUrl;
        $this->targetFolder = $targetFolder;
        $this->targetPath = $targetPath;
        $this->logger = $logger;

        if (!empty($caCert)) {
            $this->streamContext['ssl']['cafile'] = $caCert;
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector): bool
    {
        return true;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws ExceptionInterface
     */
    public function publish(DataCollectorInterface $collector): bool
    {
        $filesystem = new Filesystem();

        $sslContext = stream_context_create($this->streamContext);

        $assetSourceUrl = $this->getSourceUrl($collector);
        try {
            $source = fopen($assetSourceUrl, 'r', false, $sslContext);
        } catch (\Exception $exception) {
            throw new FilesystemException(sprintf(
                'Cannot open asset source path "%s" for reading.',
                $assetSourceUrl
            ), 500, $exception);
        }

        $assetTargetFilename = $this->getTargetFilename($collector);
        $assetTargetPath = $this->targetFolder . $this->targetPath . $assetTargetFilename;
        $assetTargetFolder = dirname($assetTargetPath);

        // Clean up - remove and create new folder
        if (file_exists($assetTargetFolder)) {
            try {
                $filesystem->remove($assetTargetFolder);
            } catch (IOExceptionInterface $exception) {
                throw new FilesystemException(sprintf(
                    'Cannot remove previous assets in folder "%s".',
                    $assetTargetFolder
                ), 500, $exception);
            }
        }
        try {
            $filesystem->mkdir($assetTargetFolder);
        } catch (IOExceptionInterface $exception) {
            throw new FilesystemException(sprintf(
                'Cannot open create asset target folder "%s".',
                $assetTargetFolder
            ), 500, $exception);
        }

        try {
            $target = fopen($assetTargetPath, 'w');
        } catch (\Exception $exception) {
            throw new FilesystemException(sprintf(
                'Cannot open asset target path "%s" for writing.',
                $assetTargetPath
            ), 500, $exception);
        }

        stream_copy_to_stream($source, $target);
        fclose($source);
        fclose($target);

        $this->updateManifestJson($collector->getItem('id')->getValue(), $assetTargetFilename);

        return true;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws ExceptionInterface
     */
    public function delete(DataCollectorInterface $collector): bool
    {
        $assetTargetPath = $this->targetFolder . $this->targetPath . $this->getTargetFilename($collector);
        if (!file_exists($assetTargetPath)) {
            $this->logger->warning(sprintf(
                'Cannot delete asset "%s": File does not exist.',
                $assetTargetPath
            ));

            return false;
        }

        $this->updateManifestJson($collector->getItem('id')->getValue(), '');

        $filesystem = new Filesystem();

        // Remove asset
        try {
            $filesystem->remove($assetTargetPath);
        } catch (IOExceptionInterface $exception) {
            throw new FilesystemException(sprintf(
                'Cannot delete asset "%s".',
                $assetTargetPath
            ), 500, $exception);
        }

        // Recursively delete folders until parent folder is not empty
        $path = $assetTargetPath;
        while (
            ($path = dirname($path)) &&
            $path !== $this->targetFolder . $this->targetPath &&
            !(new \FilesystemIterator($path))->valid()
        ) {
            try {
                $filesystem->remove($path);
            } catch (IOExceptionInterface $exception) {
                throw new FilesystemException(sprintf(
                    'Cannot delete asset folder "%s".',
                    $path
                ), 500, $exception);
            }
        }

        return true;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return string
     */
    private function getSourceUrl(DataCollectorInterface $collector): string
    {
        $path = $collector->getItem('path')->getValue();
        $filename = $collector->getItem('filename')->getValue();

        return
            $this->sourceUrl .
            implode('/', array_map('rawurlencode', explode('/', $path))) .
            rawurlencode($filename);
    }

    /**
     * @param DataCollectorInterface $collector
     * @return string
     */
    private function getTargetFilename(DataCollectorInterface $collector): string
    {
        $path = preg_replace('~(.)~', '/$1', sprintf('%010s', $collector->getItem('id')->getValue()));

        return $path . '/' . $collector->getItem('filename')->getValue();
    }

    /**
     * @param int $assetId
     * @param string $assetTargetFilename
     */
    private function updateManifestJson($assetId, $assetTargetFilename)
    {
        $manifestJson = $this->targetFolder . $this->targetPath . '/manifest.json';
        if (file_exists($manifestJson)) {
            $lines = json_decode(file_get_contents($manifestJson));
            if (!$lines instanceof \stdClass) {
                $lines = new \stdClass();
            }
        } else {
            $lines = new \stdClass();
        }

        if (0 < strlen($assetTargetFilename)) {
            $lines->$assetId = $this->targetPath . $assetTargetFilename;
        } else {
            unset($lines->$assetId);
        }

        file_put_contents($manifestJson, json_encode($lines));
    }
}
