<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\PaymentPage;

use LifeStyle\ComputopPaygateAdapter\Model\RequestModel as RequestModelBase;

/**
 * Class RequestModel
 * @package LifeStyle\ComputopPaygateAdapter\Model\PaymentPage
 */
class RequestModel extends RequestModelBase
{
    /**
     * `URLSuccess` -- Complete URL which calls up Paygate if payment has been successful. The URL may be called up only
     * via port 443. This URL may not contain parameters: In order to exchange values between Paygate and shop, please
     * use the parameter `UserData`.
     *
     * @var string
     */
    private $redirectOnSuccess;

    /**
     * `URLFailure` -- Complete URL which calls up Paygate if payment has been unsuccessful. The URL may be called up
     * only via port 443 This URL may not contain parameters: In order to exchange values between Paygate and shop,
     * please use the parameter `UserData`.
     *
     * @var string
     */
    private $redirectOnFailure;

    /**
     * `Response` -- Status response sent by Paygate to URLSuccess and URLFailure, should be encrypted.
     *
     * @var bool
     */
    private $encryptedResponse;

    /**
     * `URLNotify` -- Complete URL which Paygate calls up in order to notify the shop about the payment result. The URL
     * may be called up only via port 443. It may not contain parameters: Use the `UserData` parameter instead.
     *
     * @var string
     */
    private $notificationTarget;

    /**
     * `UserData` -- If specified at request, Paygate forwards the parameter with the payment result to the shop.
     *
     * @var string
     */
    private $userData;

    /**
     * `Custom` -- The merchant can submit several values separated by `|` which are returned unencrypted and separated by
     * `&`.
     *
     * @var string
     * @deprecated This is not officially supported across all paytype definitions.
     */
    private $customData;

    /**
     * @return string
     */
    public function getRedirectOnSuccess()
    {
        return $this->redirectOnSuccess;
    }

    /**
     * @param string $redirectOnSuccess
     * @return RequestModel
     */
    public function setRedirectOnSuccess($redirectOnSuccess)
    {
        $this->redirectOnSuccess = $redirectOnSuccess;
        return $this;
    }

    /**
     * @return string
     */
    public function getRedirectOnFailure()
    {
        return $this->redirectOnFailure;
    }

    /**
     * @param string $redirectOnFailure
     * @return RequestModel
     */
    public function setRedirectOnFailure($redirectOnFailure)
    {
        $this->redirectOnFailure = $redirectOnFailure;
        return $this;
    }

    /**
     * @return bool
     */
    public function isEncryptedResponse()
    {
        return $this->encryptedResponse;
    }

    /**
     * @param bool $encryptedResponse
     * @return RequestModel
     */
    public function setEncryptedResponse($encryptedResponse)
    {
        $this->encryptedResponse = $encryptedResponse;
        return $this;
    }

    /**
     * @return string
     */
    public function getNotificationTarget()
    {
        return $this->notificationTarget;
    }

    /**
     * @param string $notificationTarget
     * @return RequestModel
     */
    public function setNotificationTarget($notificationTarget)
    {
        $this->notificationTarget = $notificationTarget;
        return $this;
    }

    /**
     * @return string
     */
    public function getUserData()
    {
        return $this->userData;
    }

    /**
     * @param string $userData
     * @return RequestModel
     */
    public function setUserData($userData)
    {
        $this->userData = $userData;
        return $this;
    }

    /**
     * @return string
     */
    public function getCustomData()
    {
        return $this->customData;
    }

    /**
     * @param string $customData
     * @return RequestModel
     */
    public function setCustomData($customData)
    {
        $this->customData = $customData;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        $parameterData = parent::getParameterData();

        $parameterData['URLSuccess'] = $this->redirectOnSuccess;
        $parameterData['URLFailure'] = $this->redirectOnFailure;
        $parameterData['Response'] = (true === $this->encryptedResponse)
            ? 'encrypt'
            : null;
        $parameterData['URLNotify'] = $this->notificationTarget;
        $parameterData['UserData'] = $this->userData;
        $parameterData['Custom'] = $this->customData;

        return $parameterData;
    }
}
