<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\CreditCard;

use LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\RequestModel as RequestModelBase;

/**
 * Class RequestModel
 * @package LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\CreditCard
 */
class RequestModel extends RequestModelBase
{
    /**
     * `CCNr` -- Credit card number at least 12-digit, numerical without spaces. You can optionally also transmit a
     * pseudo card number (PCN).
     *
     * @var string
     */
    private $creditCardNumber;

    /**
     * `CCCVC` -- Card verification number. The last 3 (or 4) digits on the signature strip of the credit card.
     *
     * @var string
     */
    private $creditCardValidationCode;

    /**
     * `CCExpiry` -- Expiry date of the credit card in the format `YYYYMM`, e.g. 201707.
     *
     * @var string
     */
    private $creditCardExpiry;

    /**
     * `CCBrand` -- Credit card brand. Please note the spelling! According to table of credit card brands!
     *
     * @var string
     */
    private $creditCardBrand;

    /**
     * `Capture` -- Determines the type and time of capture. Possible values:
     *
     * - `AUTO`:      Capturing immediately after authorisation (default value).
     * - `MANUAL`:    Capturing made by the merchant.
     * - `{Number}`:  Delay in hours until the capture (whole number only; 1 to 696).
     *
     * @var string|int
     */
    private $capture;

    /**
     * `OrderDesc` -- Description of purchased goods, unit prices, ...
     *
     * To test error behaviour, set this to `test:0000`. The part beyond the colon can be substituted with any of the
     * codes listed under "Detailed error code".
     *
     * @var string
     */
    private $orderDescription;

    /**
     * @return string
     */
    public function getCreditCardNumber()
    {
        return $this->creditCardNumber;
    }

    /**
     * @param string $creditCardNumber
     * @return RequestModel
     */
    public function setCreditCardNumber($creditCardNumber)
    {
        $this->creditCardNumber = $creditCardNumber;
        return $this;
    }

    /**
     * @return string
     */
    public function getCreditCardValidationCode()
    {
        return $this->creditCardValidationCode;
    }

    /**
     * @param string $creditCardValidationCode
     * @return RequestModel
     */
    public function setCreditCardValidationCode($creditCardValidationCode)
    {
        $this->creditCardValidationCode = $creditCardValidationCode;
        return $this;
    }

    /**
     * @return string
     */
    public function getCreditCardExpiry()
    {
        return $this->creditCardExpiry;
    }

    /**
     * @param string $creditCardExpiry
     * @return RequestModel
     */
    public function setCreditCardExpiry($creditCardExpiry)
    {
        $this->creditCardExpiry = $creditCardExpiry;
        return $this;
    }

    /**
     * @return string
     */
    public function getCreditCardBrand()
    {
        return $this->creditCardBrand;
    }

    /**
     * @param string $creditCardBrand
     * @return RequestModel
     */
    public function setCreditCardBrand($creditCardBrand)
    {
        $this->creditCardBrand = $creditCardBrand;
        return $this;
    }

    /**
     * @return int|string
     */
    public function getCapture()
    {
        return $this->capture;
    }

    /**
     * @param int|string $capture
     * @return RequestModel
     */
    public function setCapture($capture)
    {
        $this->capture = $capture;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderDescription()
    {
        return $this->orderDescription;
    }

    /**
     * @param string $orderDescription
     * @return RequestModel
     */
    public function setOrderDescription($orderDescription)
    {
        $this->orderDescription = $orderDescription;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        $parameterData = parent::getParameterData();

        $parameterData['CCNr'] = $this->creditCardNumber;
        $parameterData['CCCVC'] = $this->creditCardValidationCode;
        $parameterData['CCExpiry'] = $this->creditCardExpiry;
        $parameterData['CCBrand'] = $this->creditCardBrand;
        $parameterData['Capture'] = $this->capture;
        $parameterData['OrderDesc'] = $this->orderDescription;

        return $parameterData;
    }
}
