# computop-paygate-adapter

This library can be used to create the data required to use the computop paygate (payment-page and server-to-server).

## Installation

```bash
composer require lifestyle/computop-paygate-adapter
```

> Note: Make sure to add the lifestyle package repository to the `composer.json` file of the project like this:
> ```json
> {
>   // ...
>   "repositories": [
>     {
>       "type": "composer",
>       "url": "https://composer.life-style.de/"
>     }
>   ],
>   // ...
> }
> ```
> Note: Don't forget to add the `auth.json` file to the project root (besides the `composer.json` file).

> Note: While development, use the following configuration to work on this library and the project simultaneously:
> ```json
> {
>   // ...
>   "repositories": [
>     {
>       "type": "git",
>       "url": "./../computop-paygate-adapter"
>     }
>   ],
>   // ...
> }
> ```

## Usage

See `example/` for a bunch of example scenarios.

### Usage with 3D-Secure 1.0/2.0

The model fully supports the credit card 3DS (as long as it is configured and activated for the merchant). Please note,
that the 3DS fields are embedded into the normal credit card model classes, but to activate 3DS 2.0, you must set the
`MsgVer` field to the value `2.0` (read
[the reference](https://developer.computop.com/display/public/DOCCT/EMV+3-D+Secure+EN#EMV3-DSecureEN-MessageVersion2)
here):

```php
<?php

use LifeStyle\ComputopPaygateAdapter\Model\PaymentPage\CreditCard\RequestModel;

$model = new RequestModel();
$model->setMessageVersion('2.0');

// ...

```

To not use 3D-Secure, simply omit its fields.

> **Open TODO**:
>
> * Create response model for 3DS json objects

## Documentation

### Paytype documentation and field information

The definition for the short-name field description can be found here:

https://developer.computop.com/display/public/DOCCT/Definition+EN

* DirectDebit

  https://developer.computop.com/display/public/DOCCT/Direct+debit+EN#DirectdebitEN-Generalparametersoftheinterface

  https://developer.computop.com/display/public/DOCCT/Direct+debit+EN#DirectdebitEN-Callofinterface:generalparameters

* CreditCard

  https://developer.computop.com/display/public/DOCCT/Card+processing+EN#CardprocessingEN-Generalparametersoftheinterface

  https://developer.computop.com/display/public/DOCCT/Card+processing+EN#CardprocessingEN-Callofinterface:generalparameters

* Klarna Sofort

  https://developer.computop.com/display/public/DOCCT/Klarna+Sofort+EN#KlarnaSofortEN-Callingtheinterface

* *To be continued...*

The definition for 3D-Secure 2.0 json objects and fields can be found here:

* https://developer.computop.com/display/public/DOCCT/JSON+Objects+EN

### Paytype handler

A paytype handler must implement `LifeStyle\ComputopPaygateAdapter\Paytype\PaytypeHandlerInterface` or extend
`LifeStyle\ComputopPaygateAdapter\Paytype\PaytypeHandlerAbstract` depending on the use-case. A rule of thumb is to use
the interface only if the bahviour of the paytype handler should differ from the default process defined inside of the
abstract class.

In any case, to use the newly created paytype handler, add it to the `LifeStyle\ComputopPaygateAdapter\PaygateService`
instance. Internally a slightly modified version of the
[strategy pattern](https://designpatternsphp.readthedocs.io/en/latest/Behavioral/Strategy/README.html) is used.

Here is an example of a basic paytype handler extending the abstract class:

```php
<?php

namespace LifeStyle\ComputopPaygateAdapter\Paytype\PaymentPage;

use LifeStyle\ComputopPaygateAdapter\Paytype\PaytypeHandlerAbstract;

class CustomPaytypeHandler extends PaytypeHandlerAbstract
{
    const PAYTYPE_NAME = '<paytype-identifier>';

    protected static $TARGET = 'https://www.computop-paygate.com/<paytype-target>.aspx';
}

```

The second step is to create a dedicated class for the paytype of `LifeStyle\ComputopPaygateAdapter\Model\RequestModel`
which holds all information and fields required/available.

Make sure to add the correct mapping for each field by setting additional fields inside the `getParameterData()` method
of that model.

### Paytype model extension

In some cases you might want to add or remove fields from a given request model.

To achieve that, create a new class that extends the desire base request model.

Here is an example on how rename the `OrderDesc` field of the credit card model to one specific to an acquirer:

> Since there is no `OrderDesc` field described for the general parameters in the documentation, this could be a common
> use-case.

```php
<?php

namespace LifeStyle\ComputopPaygateAdapter\Model\PaymentPage\CreditCard\Extension;

use LifeStyle\ComputopPaygateAdapter\Model\PaymentPage\CreditCard\RequestModel as BaseRequestModel;

class SpecificRequestModel extends BaseRequestModel
{
    /**
     * @var string|null
     */
    private $orderDescriptionFn = null;

    /**
     * @param string|null $fieldName
     * @return SpecificRequestModel
     */
    public function setOrderDescriptionFn($fieldName)
    {
        $this->orderDescriptionFn = $fieldName;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        $parameterData = parent::getParameterData();

        if (null !== $this->orderDescriptionFn && 'OrderDesc' !== $this->orderDescriptionFn) {
            $parameterData[$this->orderDescriptionFn] = $parameterData['OrderDesc'];
            unset($parameterData['OrderDesc']);
        }

        return $parameterData;
    }
}

```

### Paytype error-code testing

While the merchant account is in model-mode, the paygate can be manipulated to return a certain error code. Depending on
what paytype is used for testing, the model should contain a field called `orderDescription`.

It **must** be set to a value of `model:0000` to proceed with successful payments during testing. The part beyond the
colon can be substituted with any of the codes listed under *detailed error codes* found
[here](https://developer.computop.com/display/public/DOCCT/A4+Error+codes#A4Errorcodes-Digits5..8:Detailederrorcode).

### Testing Server-to-server connection (with Postman)

To model the computop paygate with the server-to-server request use the following configuration in postman:

`POST $target`, where `$target` is the target url from the `LifeStyle\ComputopPaygateAdapter\Model\ResponseModel`
object.

Then set a body of type `x-www-form-urlencoded` with the key-value data from the
`LifeStyle\ComputopPaygateAdapter\Model\ResponseModel` object. Typically, these contain the following fields:

```
MerchantID=<MERCHANT>
Data=d83d7c6d7394e...
Len=282
```

## License

Proprietary.
