<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\PaymentPage\CreditCard;

use LifeStyle\ComputopPaygateAdapter\Model\PaymentPage\RequestModel as RequestModelBase;
use LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo\Address;
use LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo\CredentialOnFile;
use LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo\CustomerInfo;
use LifeStyle\ComputopPaygateAdapter\Utility\ParameterDataEncodeTrait;

/**
 * Class RequestModel
 * @package LifeStyle\ComputopPaygateAdapter\Model\PaymentPage\CreditCard
 */
class RequestModel extends RequestModelBase
{
    use ParameterDataEncodeTrait;

    /**
     * `Amount3D` -- Only for 3DSecure: Amount for authentication with Verified, SecureCode and SafeKey if Amount
     * deviates. Amount in the smallest currency unit (e.g. EUR Cent).
     *
     * @var int
     */
    private $amount3d;

    /**
     * `OrderDesc` -- Description of purchased goods, unit prices, ...
     *
     * To test error behaviour, set this to `test:0000`. The part beyond the colon can be substituted with any of the
     * codes listed under "Detailed error code".
     *
     * @var string
     */
    private $orderDescription;

    /**
     * `MsgVer` --
     *
     * **Message Version 2**
     *
     * To handle the amount of additional non-payment data and to maintain downward compatibility as much as possible
     * Computop decided to version its Paygate card interface via the additional data element MsgVer. The upgraded API
     * is still based on key-value pairs but relies heavily on Base64 encoded JSON objects to aid readability and
     * client-side scripting.
     *
     * Merchants will still be able to use our classic interface for requests even with 3DS 2.0 but there are some
     * limitations:
     *
     * - Many additional data points for issuer risk analysis are not available and thus, the frictionless ratio might
     *  be lower.
     * - API responses and notifications do include new JSON objects to cater for 3DS 2.0 protocol specifications and
     *  require modification of existing merchant integrations.
     *
     * For these reasons it is highly recommended to upgrade to version 2.
     *
     * To enable that, set this field to the value `2.0`.
     *
     * @var string
     */
    private $messageVersion;

    /**
     * @var CustomerInfo
     */
    private $billToCustomer;

    /**
     * @var Address
     */
    private $billingAddress;

    /**
     * @var CustomerInfo
     */
    private $shipToCustomer;

    /**
     * @var Address
     */
    private $shippingAddress;

    /**
     * @var CredentialOnFile
     */
    private $credentialOnFile;

    /**
     * @return int
     */
    public function getAmount3D()
    {
        return $this->amount3d;
    }

    /**
     * @param int $amount3d
     * @return RequestModel
     */
    public function setAmount3D($amount3d)
    {
        $this->amount3d = $amount3d;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderDescription()
    {
        return $this->orderDescription;
    }

    /**
     * @return CustomerInfo
     */
    public function getBillToCustomer()
    {
        return $this->billToCustomer;
    }

    /**
     * @param CustomerInfo $billToCustomer
     * @return RequestModel
     */
    public function setBillToCustomer($billToCustomer)
    {
        $this->billToCustomer = $billToCustomer;
        return $this;
    }

    /**
     * @return Address
     */
    public function getBillingAddress()
    {
        return $this->billingAddress;
    }

    /**
     * @param Address $billingAddress
     * @return RequestModel
     */
    public function setBillingAddress($billingAddress)
    {
        $this->billingAddress = $billingAddress;
        return $this;
    }

    /**
     * @return CredentialOnFile
     */
    public function getCredentialOnFile()
    {
        return $this->credentialOnFile;
    }

    /**
     * @param CredentialOnFile $credentialOnFile
     * @return RequestModel
     */
    public function setCredentialOnFile($credentialOnFile)
    {
        $this->credentialOnFile = $credentialOnFile;
        return $this;
    }

    /**
     * @return string
     */
    public function getMessageVersion()
    {
        return $this->messageVersion;
    }

    /**
     * @param string $messageVersion
     * @return RequestModel
     */
    public function setMessageVersion($messageVersion)
    {
        $this->messageVersion = $messageVersion;
        return $this;
    }

    /**
     * @return CustomerInfo
     */
    public function getShipToCustomer()
    {
        return $this->shipToCustomer;
    }

    /**
     * @param CustomerInfo $shipToCustomer
     * @return RequestModel
     */
    public function setShipToCustomer($shipToCustomer)
    {
        $this->shipToCustomer = $shipToCustomer;
        return $this;
    }

    /**
     * @return Address
     */
    public function getShippingAddress()
    {
        return $this->shippingAddress;
    }

    /**
     * @param Address $shippingAddress
     * @return RequestModel
     */
    public function setShippingAddress($shippingAddress)
    {
        $this->shippingAddress = $shippingAddress;
        return $this;
    }

    /**
     * @param string $orderDescription
     * @return RequestModel
     */
    public function setOrderDescription($orderDescription)
    {
        $this->orderDescription = $orderDescription;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        $parameterData = parent::getParameterData();

        $parameterData['Amount3D'] = $this->amount3d;
        $parameterData['OrderDesc'] = $this->orderDescription;

        $parameterData['MsgVer'] = $this->messageVersion;

        $parameterData['credentialOnFile'] = !empty($this->credentialOnFil)
            ? $this->encode($this->credentialOnFile)
            : null;

        $parameterData['billToCustomer'] = !empty($this->billToCustomer)
            ? $this->encode($this->billToCustomer)
            : null;

        $parameterData['billingAddress'] = !empty($this->billingAddress)
            ? $this->encode($this->billingAddress)
            : null;

        $parameterData['shipToCustomer'] = !empty($this->shipToCustomer)
            ? $this->encode($this->shipToCustomer)
            : null;

        $parameterData['shippingAddress'] = !empty($this->shippingAddress)
            ? $this->encode($this->shippingAddress)
            : null;

        return $parameterData;
    }
}
