<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\CreditCard\GICC;

use LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\CreditCard\RequestModel as RequestModelBase;

/**
 * Class RequestModel
 *
 * **WARNING**: The model is specific for _GICC_ acquirers!
 *
 * @package LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\CreditCard\GICC
 */
class RequestModel extends RequestModelBase
{
    /**
     * `RefNr` -- Merchant’s unique reference number, which serves as payout reference in the acquirer EPA file. Please
     * note, without the own shop reference delivery you cannot read out the EPA transaction and regarding the
     * additional CompuTop settlement file (CTSF) we cannot add the additional payment data.
     *
     * If `RefNr` is not submitted, the value of `TransID` will be used.
     *
     * @var string
     */
    private $referenceNumber;

    /**
     * `AddrCountryCode` -- Country code according to ISO-3166-1 (e.g. `DE`). It can be transmitted with two or three
     * digits optionally.
     *
     * @var string
     */
    private $addressCountryCode;

    /**
     * `RTF` -- Flag for repeated payments (subscriptions). Possible values:
     *
     * - `I`: Initial payment of a new subscription.
     * - `R`: Routine payment (recurring).
     * - `E`: Establishment of Credential on File Agreement.
     * - `C`: Unscheduled Cardholder initiated transaction using stored Credential on File.
     * - `M`: Unscheduled Merchant initiated transaction using stored Credential on File.
     *
     * @var string
     */
    private $retailFlag;

    /**
     * `ChDesc` -- Text printed on the customer's credit card bill alongside the payee.
     *
     * @var string
     */
    private $cardHolderDescription;

    /**
     * `AccVerify` -- If `AccVerify` = `yes` (e.g. this field is set to true) the card will be checked at the acquirer
     * according to the acquirer's interface description. The merchant has to submit only this parameter, the parameter
     * `Amount` is optional. If `Amount` is used we replace the amount according to acquirer’s interface description. At
     * payment always `Amount` = `0` is stored. Possible value:
     *
     * - `yes`: Verify the card.
     *
     * @var bool
     */
    private $accountVerification;

    /**
     * `CountryCode` -- Card holder information: Country.
     *
     * @var string
     */
    private $cardHolderCountryCode;

    /**
     * `Textfeld1` -- Card holder information: Name.
     *
     * @var string
     */
    private $cardHolderName;

    /**
     * `Textfeld2` -- Card holder information: City.
     *
     * @var string
     */
    private $cardHolderCity;

    /**
     * @return string
     */
    public function getReferenceNumber()
    {
        return $this->referenceNumber;
    }

    /**
     * @param string $referenceNumber
     * @return RequestModel
     */
    public function setReferenceNumber($referenceNumber)
    {
        $this->referenceNumber = $referenceNumber;
        return $this;
    }

    /**
     * @return string
     */
    public function getAddressCountryCode()
    {
        return $this->addressCountryCode;
    }

    /**
     * @param string $addressCountryCode
     * @return RequestModel
     */
    public function setAddressCountryCode($addressCountryCode)
    {
        $this->addressCountryCode = $addressCountryCode;
        return $this;
    }

    /**
     * @return string
     */
    public function getRetailFlag()
    {
        return $this->retailFlag;
    }

    /**
     * @param string $retailFlag
     * @return RequestModel
     */
    public function setRetailFlag($retailFlag)
    {
        $this->retailFlag = $retailFlag;
        return $this;
    }

    /**
     * @return string
     */
    public function getCardHolderDescription()
    {
        return $this->cardHolderDescription;
    }

    /**
     * @param string $cardHolderDescription
     * @return RequestModel
     */
    public function setCardHolderDescription($cardHolderDescription)
    {
        $this->cardHolderDescription = $cardHolderDescription;
        return $this;
    }

    /**
     * @return bool
     */
    public function isAccountVerification()
    {
        return $this->accountVerification;
    }

    /**
     * @param bool $accountVerification
     * @return RequestModel
     */
    public function setAccountVerification($accountVerification)
    {
        $this->accountVerification = $accountVerification;
        return $this;
    }

    /**
     * @return string
     */
    public function getCardHolderCountryCode()
    {
        return $this->cardHolderCountryCode;
    }

    /**
     * @param string $cardHolderCountryCode
     * @return RequestModel
     */
    public function setCardHolderCountryCode($cardHolderCountryCode)
    {
        $this->cardHolderCountryCode = $cardHolderCountryCode;
        return $this;
    }

    /**
     * @return string
     */
    public function getCardHolderName()
    {
        return $this->cardHolderName;
    }

    /**
     * @param string $cardHolderName
     * @return RequestModel
     */
    public function setCardHolderName($cardHolderName)
    {
        $this->cardHolderName = $cardHolderName;
        return $this;
    }

    /**
     * @return string
     */
    public function getCardHolderCity()
    {
        return $this->cardHolderCity;
    }

    /**
     * @param string $cardHolderCity
     * @return RequestModel
     */
    public function setCardHolderCity($cardHolderCity)
    {
        $this->cardHolderCity = $cardHolderCity;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        $parameterData = parent::getParameterData();

        // Already set:
        //$parameterData['OrderDesc'];

        $parameterData['RefNr'] = $this->referenceNumber;
        $parameterData['AddrCountryCode'] = $this->addressCountryCode;
        $parameterData['RTF'] = $this->retailFlag;
        $parameterData['ChDesc'] = $this->cardHolderDescription;

        if (isset($this->accountVerification)) {
            $parameterData['AccVerify'] = (true === $this->accountVerification)
                ? 'yes'
                : null;

            if (null !== ($parameterData['AccVerify'])) {
                unset($parameterData['Amount']);
            }
        }

        $parameterData['CountryCode'] = $this->cardHolderCountryCode;
        $parameterData['Textfeld1'] = $this->cardHolderName;
        $parameterData['Textfeld2'] = $this->cardHolderCity;

        return $parameterData;
    }
}
