<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model;

/**
 * Class RequestModel
 * @package LifeStyle\ComputopPaygateAdapter\Model
 */
class RequestModel implements ParameterProviderInterface
{
    /**
     * The paytype request target base. This is should normally be set by the paytype handler.
     *
     * @var string
     */
    private $target;

    /**
     * An id assigned by Paygate for the payment. This will be supplied with the response and is needed for further
     * requests.
     *
     * @var string
     * @deprecated Is not set in the parameter data by default!
     */
    private $paymentId;

    /**
     * `MerchantID` -- MerchantID, assigned by Computop. Additionally this parameter has to be passed in plain language
     * too.
     *
     * @var string
     */
    private $merchantId;

    /**
     * `TransID` -- TransactionID which should be unique for each payment.
     *
     * @var string
     */
    private $transactionId;

    /**
     * `Amount` -- Amount in the smallest currency unit (e.g. EUR Cent).
     *
     * @var int
     */
    private $amount;

    /**
     * `Currency` -- Currency, three digits DIN / ISO 4217 Only EUR is permissible.
     *
     * @var string
     */
    private $currency;

    /**
     * `MAC` -- Hash Message Authentication Code (HMAC) with SHA-256 algorithm.
     *
     * @var string
     * @deprecated This is generated automatically.
     */
    private $mac;

    /**
     * `ReqID` -- To avoid double payments, enter an alphanumeric value which identifies your transaction and may be
     * assigned only once. If the transaction is submitted again with the same `ReqID`, the Paygate will not carry out
     * the payment, but will just return the status of the original transaction. Please note that the Paygate must have
     * a finalized transaction status for the first initial action. Submissions with identical `ReqID` for an open
     * status will be processed regularly.
     *
     * @var string
     */
    private $requestId;

    /**
     * @return string
     */
    public function getTarget()
    {
        return $this->target;
    }

    /**
     * @param string $target
     * @return RequestModel
     */
    public function setTarget($target)
    {
        $this->target = $target;
        return $this;
    }

    /**
     * @return string
     * @deprecated
     */
    public function getPaymentId()
    {
        return $this->paymentId;
    }

    /**
     * @param string $paymentId
     * @return RequestModel
     * @deprecated
     */
    public function setPaymentId($paymentId)
    {
        $this->paymentId = $paymentId;
        return $this;
    }

    /**
     * @return string
     */
    public function getMerchantId()
    {
        return $this->merchantId;
    }

    /**
     * @param string $merchantId
     * @return RequestModel
     */
    public function setMerchantId($merchantId)
    {
        $this->merchantId = $merchantId;
        return $this;
    }

    /**
     * @return string
     */
    public function getTransactionId()
    {
        return $this->transactionId;
    }

    /**
     * @param string $transactionId
     * @return RequestModel
     */
    public function setTransactionId($transactionId)
    {
        $this->transactionId = $transactionId;
        return $this;
    }

    /**
     * @return int
     */
    public function getAmount()
    {
        return $this->amount;
    }

    /**
     * @param int $amount
     * @return RequestModel
     */
    public function setAmount($amount)
    {
        $this->amount = $amount;
        return $this;
    }

    /**
     * @return string
     */
    public function getCurrency()
    {
        return $this->currency;
    }

    /**
     * @param string $currency
     * @return RequestModel
     */
    public function setCurrency($currency)
    {
        $this->currency = $currency;
        return $this;
    }

    /**
     * @return string
     */
    public function getMac()
    {
        return $this->mac;
    }

    /**
     * @param string $mac
     * @return RequestModel
     */
    public function setMac($mac)
    {
        $this->mac = $mac;
        return $this;
    }

    /**
     * @return string
     */
    public function getRequestId()
    {
        return $this->requestId;
    }

    /**
     * @param string $requestId
     * @return RequestModel
     */
    public function setRequestId($requestId)
    {
        $this->requestId = $requestId;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        return [
            'MerchantID'    => $this->merchantId,
            'TransID'       => $this->transactionId,
            'Amount'        => $this->amount,
            'Currency'      => $this->currency,
            'MAC'           => $this->mac,
            'ReqID'         => $this->requestId,
        ];
    }
}
