<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo;

use JsonSerializable;

/**
 * Class Card
 * @package LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo
 *
 * @see https://developer.computop.com/display/public/DOCCT/card%3Arequest+EN
 */
class Card implements JsonSerializable
{
    /**
     * `securityCode` -- Card security value.
     *
     * Optional field.
     *
     * @var string
     */
    private $securityCode;

    /**
     * `expiryDate` -- Card expiry in format YYYYMM.
     *
     * Mandatory field.
     *
     * @var string
     */
    private $expiryDate;

    /**
     * `startDate` -- Card start in format YYYYMM (only applicable to some UK debit cards).
     *
     * Conditional field -> only for some UK cards.
     *
     * @var string
     */
    private $startDate;

    /**
     * `cardholderName` -- The name of the cardholder as printed on the card. Alphanumeric special characters,
     * listed in EMV Book 4, “Appendix B”.
     *
     * Mandatory field.
     *
     * @var string
     */
    private $cardholderName;

    /**
     * `issueNumber` -- Issue number of the card (only applicable to some UK debit cards).
     *
     * Conditional field -> only for some UK cards.
     *
     * @var string
     */
    private $issueNumber;

    /**
     * `number` -- Card number or card token. Can also be the PKN.
     *
     * Mandatory field.
     *
     * @var string
     */
    private $number;

    /**
     * `brand` -- Card network. Examples -> VISA, MasterCard, AMEX, etc.
     *
     * Mandatory field.
     *
     * @var string
     */
    private $brand;

    /**
     * @return string
     */
    public function getSecurityCode()
    {
        return $this->securityCode;
    }

    /**
     * @param string $securityCode
     * @return Card
     */
    public function setSecurityCode($securityCode)
    {
        $this->securityCode = $securityCode;
        return $this;
    }

    /**
     * @return string
     */
    public function getExpiryDate()
    {
        return $this->expiryDate;
    }

    /**
     * @param string $expiryDate
     * @return Card
     */
    public function setExpiryDate($expiryDate)
    {
        $this->expiryDate = $expiryDate;
        return $this;
    }

    /**
     * @return string
     */
    public function getStartDate()
    {
        return $this->startDate;
    }

    /**
     * @param string $startDate
     * @return Card
     */
    public function setStartDate($startDate)
    {
        $this->startDate = $startDate;
        return $this;
    }

    /**
     * @return string
     */
    public function getCardholderName()
    {
        return $this->cardholderName;
    }

    /**
     * @param string $cardholderName
     * @return Card
     */
    public function setCardholderName($cardholderName)
    {
        $this->cardholderName = $cardholderName;
        return $this;
    }

    /**
     * @return string
     */
    public function getIssueNumber()
    {
        return $this->issueNumber;
    }

    /**
     * @param string $issueNumber
     * @return Card
     */
    public function setIssueNumber($issueNumber)
    {
        $this->issueNumber = $issueNumber;
        return $this;
    }

    /**
     * @return string
     */
    public function getNumber()
    {
        return $this->number;
    }

    /**
     * @param string $number
     * @return Card
     */
    public function setNumber($number)
    {
        $this->number = $number;
        return $this;
    }

    /**
     * @return string
     */
    public function getBrand()
    {
        return $this->brand;
    }

    /**
     * @param string $brand
     * @return Card
     */
    public function setBrand($brand)
    {
        $this->brand = $brand;
        return $this;
    }

    /**
     * @inheritDoc
     * @internal
     */
    public function jsonSerialize()
    {
        return array_filter([
            'securityCode' => $this->securityCode,
            'expiryDate' => $this->expiryDate,
            'startDate' => $this->startDate,
            'cardholderName' => $this->cardholderName,
            'issueNumber' => $this->issueNumber,
            'number' => $this->number,
            'brand' => $this->brand,
        ]);
    }
}
