<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\DirectDebit;

use LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\RequestModel as RequestModelBase;

/**
 * Class RequestModel
 * @package LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\DirectDebit
 */
class RequestModel extends RequestModelBase
{
    /**
     * `AccBank` -- Bank name.
     *
     * @var string
     */
    private $accountBank;

    /**
     * `AccOwner` -- Account holder.
     *
     * @var string
     */
    private $accountOwner;

    /**
     * `Capture` -- Determines the type and time of capture. Possible values:
     *
     * - `AUTO`:      Capturing immediately after authorisation (default value).
     * - `MANUAL`:    Capturing made by the merchant.
     * - `{Number}`:  Delay in hours until the capture (whole number only; 1 to 696).
     *
     * @var string|int
     */
    private $capture;

    /**
     * `OrderDesc` -- Description of purchased goods, unit prices, ...
     *
     * To test error behaviour, set this to `test:0000`. The part beyond the colon can be substituted with any of the
     * codes listed under "Detailed error code".
     *
     * @var string
     */
    private $orderDescription;

    /**
     * `IBAN` -- International Bank Account Number.
     *
     * @var string
     */
    private $iban;

    /**
     * `BIC` -- Bank Identifier Code.
     *
     * @var string
     */
    private $bic;

    /**
     * `UserData` -- If specified at request, Paygate forwards the parameter with the payment result to the shop.
     *
     * @var string
     */
    private $userData;

    /**
     * `RefNr` -- Merchant’s unique reference number, which serves as payout reference in the acquirer EPA file. Please
     * note, without the own shop reference delivery you cannot read out the EPA transaction and regarding the
     * additional CompuTop settlement file (CTSF) we cannot add the additional payment data.
     *
     * If `RefNr` is not submitted, the value of `TransID` will be used.
     *
     * @var string
     */
    private $referenceNumber;

    /**
     * `OrderDesc2` -- 2. Goods description which appears on the account extract (27 characters). You can view the full
     * data in CompuTop analytics.
     *
     * @var string
     */
    private $orderDescription2;

    /**
     * `MandateID` -- For SEPA: SEPA mandate number (compulsory in the case of SEPA).
     *
     * Should be unique, is not case-sensitive.
     *
     * Permissible characters: numbers 0-9, upper case letters A-Z, lower case letters a-z, apostrophe, colon, question
     * mark, comma, full stop, plus sign, minus sign, forward slash, left and right bracket.
     *
     * @var string
     */
    private $mandateId;

    /**
     * `DtOfSgntr` -- For SEPA: Date of issuing the mandate in the format `DD.MM.YYYY` (obligatory when transferring the
     * `MandateID`).
     *
     * @var string
     */
    private $dateOfSignature;

    /**
     * `MdtSeqType` -- For SEPA: Indicates whether the direct debit is a first, recurring, one-off or final direct
     * debit. Permissible values:
     *
     * - FRST
     * - RCUR
     * - OOFF
     * - FNAL
     *
     * Standard: OOFF (single direct debit).
     *
     * @var string
     */
    private $mandateSequenceType;

    /**
     * @return string
     */
    public function getAccountBank()
    {
        return $this->accountBank;
    }

    /**
     * @param string $accountBank
     * @return RequestModel
     */
    public function setAccountBank($accountBank)
    {
        $this->accountBank = $accountBank;
        return $this;
    }

    /**
     * @return string
     */
    public function getAccountOwner()
    {
        return $this->accountOwner;
    }

    /**
     * @param string $accountOwner
     * @return RequestModel
     */
    public function setAccountOwner($accountOwner)
    {
        $this->accountOwner = $accountOwner;
        return $this;
    }

    /**
     * @return int|string
     */
    public function getCapture()
    {
        return $this->capture;
    }

    /**
     * @param int|string $capture
     * @return RequestModel
     */
    public function setCapture($capture)
    {
        $this->capture = $capture;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderDescription()
    {
        return $this->orderDescription;
    }

    /**
     * @param string $orderDescription
     * @return RequestModel
     */
    public function setOrderDescription($orderDescription)
    {
        $this->orderDescription = $orderDescription;
        return $this;
    }

    /**
     * @return string
     */
    public function getIban()
    {
        return $this->iban;
    }

    /**
     * @param string $iban
     * @return RequestModel
     */
    public function setIban($iban)
    {
        $this->iban = $iban;
        return $this;
    }

    /**
     * @return string
     */
    public function getBic()
    {
        return $this->bic;
    }

    /**
     * @param string $bic
     * @return RequestModel
     */
    public function setBic($bic)
    {
        $this->bic = $bic;
        return $this;
    }

    /**
     * @return string
     */
    public function getUserData()
    {
        return $this->userData;
    }

    /**
     * @param string $userData
     * @return RequestModel
     */
    public function setUserData($userData)
    {
        $this->userData = $userData;
        return $this;
    }

    /**
     * @return string
     */
    public function getReferenceNumber()
    {
        return $this->referenceNumber;
    }

    /**
     * @param string $referenceNumber
     * @return RequestModel
     */
    public function setReferenceNumber($referenceNumber)
    {
        $this->referenceNumber = $referenceNumber;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderDescription2()
    {
        return $this->orderDescription2;
    }

    /**
     * @param string $orderDescription2
     * @return RequestModel
     */
    public function setOrderDescription2($orderDescription2)
    {
        $this->orderDescription2 = $orderDescription2;
        return $this;
    }

    /**
     * @return string
     */
    public function getMandateId()
    {
        return $this->mandateId;
    }

    /**
     * @param string $mandateId
     * @return RequestModel
     */
    public function setMandateId($mandateId)
    {
        $this->mandateId = $mandateId;
        return $this;
    }

    /**
     * @return string
     */
    public function getDateOfSignature()
    {
        return $this->dateOfSignature;
    }

    /**
     * @param string $dateOfSignature
     * @return RequestModel
     */
    public function setDateOfSignature($dateOfSignature)
    {
        $this->dateOfSignature = $dateOfSignature;
        return $this;
    }

    /**
     * @return string
     */
    public function getMandateSequenceType()
    {
        return $this->mandateSequenceType;
    }

    /**
     * @param string $mandateSequenceType
     * @return RequestModel
     */
    public function setMandateSequenceType($mandateSequenceType)
    {
        $this->mandateSequenceType = $mandateSequenceType;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        $parameterData = parent::getParameterData();

        $parameterData['AccBank'] = $this->accountBank;
        $parameterData['AccOwner'] = $this->accountOwner;
        $parameterData['Capture'] = $this->capture;
        $parameterData['OrderDesc'] = $this->orderDescription;
        $parameterData['IBAN'] = $this->iban;
        $parameterData['BIC'] = $this->bic;
        $parameterData['UserData'] = $this->userData;

        $parameterData['RefNr'] = $this->referenceNumber;
        $parameterData['OrderDesc2'] = $this->orderDescription2;

        $parameterData['PayId'] = $this->getPaymentId();

        $parameterData['MandateID'] = $this->mandateId;
        $parameterData['DtOfSgntr'] = $this->dateOfSignature;
        $parameterData['MdtSeqType'] = $this->mandateSequenceType;

        return $parameterData;
    }
}
