<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\CreditCard;

use LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\RequestModel as RequestModelBase;
use LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo\Card;
use LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo\CredentialOnFile;
use LifeStyle\ComputopPaygateAdapter\Utility\ParameterDataEncodeTrait;

/**
 * Class RequestModel
 * @package LifeStyle\ComputopPaygateAdapter\Model\ServerToServer\CreditCard
 */
class RequestModel extends RequestModelBase
{
    use ParameterDataEncodeTrait;

    /**
     * `CCNr` -- Credit card number at least 12-digit, numerical without spaces. You can optionally also transmit a
     * pseudo card number (PCN).
     *
     * @var string
     */
    private $creditCardNumber;

    /**
     * `CCCVC` -- Card verification number. The last 3 (or 4) digits on the signature strip of the credit card.
     *
     * @var string
     */
    private $creditCardValidationCode;

    /**
     * `CCExpiry` -- Expiry date of the credit card in the format `YYYYMM`, e.g. 201707.
     *
     * @var string
     */
    private $creditCardExpiry;

    /**
     * `CCBrand` -- Credit card brand. Please note the spelling! According to table of credit card brands!
     *
     * @var string
     */
    private $creditCardBrand;

    /**
     * `Capture` -- Determines the type and time of capture. Possible values:
     *
     * - `AUTO`:      Capturing immediately after authorisation (default value).
     * - `MANUAL`:    Capturing made by the merchant.
     * - `{Number}`:  Delay in hours until the capture (whole number only; 1 to 696).
     *
     * @var string|int
     */
    private $capture;

    /**
     * `OrderDesc` -- Description of purchased goods, unit prices, ...
     *
     * To test error behaviour, set this to `test:0000`. The part beyond the colon can be substituted with any of the
     * codes listed under "Detailed error code".
     *
     * @var string
     */
    private $orderDescription;

    /**
     * `MsgVer` --
     *
     * **Message Version 2**
     *
     * To handle the amount of additional non-payment data and to maintain downward compatibility as much as possible
     * Computop decided to version its Paygate card interface via the additional data element MsgVer. The upgraded API
     * is still based on key-value pairs but relies heavily on Base64 encoded JSON objects to aid readability and
     * client-side scripting.
     *
     * Merchants will still be able to use our classic interface for requests even with 3DS 2.0 but there are some
     * limitations:
     *
     * - Many additional data points for issuer risk analysis are not available and thus, the frictionless ratio might
     *  be lower.
     * - API responses and notifications do include new JSON objects to cater for 3DS 2.0 protocol specifications and
     *  require modification of existing merchant integrations.
     *
     * For these reasons it is highly recommended to upgrade to version 2.
     *
     * To enable that, set this field to the value `2.0`.
     *
     * @var string
     */
    private $messageVersion;

    /**
     * `credentialOnFile` -- some further information about the payment which allows recurring or manually initiated
     * transactions.
     *
     * @var CredentialOnFile
     *
     * @see https://developer.computop.com/display/public/DOCCT/credentialOnFile+EN
     */
    private $credentialOnFile;

    /**
     * `card` -- Card request data.
     *
     * @var Card
     *
     * @see https://developer.computop.com/display/public/DOCCT/card%3Arequest+EN
     */
    private $card;

    /**
     * `schemeReferenceId` -- This is a unique transaction identifier provided directly from the card schemes like VISA
     * and MC in order to uniquely reference a transaction in the whole payment ecosystem. It was introduced initially
     * by VISA in accordance to their Framework specifications like COF (Credential On File) and MIT
     * (Merchant Initiated Transactions), relevant to use cases involving transaction types such as Recurring,
     * UCOF (MIT), Incremental, Delayed Authorization, Resubmission etc.
     *
     * @var string
     */
    private $schemeReferenceId;

    /**
     * @return string
     */
    public function getCreditCardNumber()
    {
        return $this->creditCardNumber;
    }

    /**
     * @param string $creditCardNumber
     * @return RequestModel
     */
    public function setCreditCardNumber($creditCardNumber)
    {
        $this->creditCardNumber = $creditCardNumber;
        return $this;
    }

    /**
     * @return string
     */
    public function getCreditCardValidationCode()
    {
        return $this->creditCardValidationCode;
    }

    /**
     * @param string $creditCardValidationCode
     * @return RequestModel
     */
    public function setCreditCardValidationCode($creditCardValidationCode)
    {
        $this->creditCardValidationCode = $creditCardValidationCode;
        return $this;
    }

    /**
     * @return string
     */
    public function getCreditCardExpiry()
    {
        return $this->creditCardExpiry;
    }

    /**
     * @param string $creditCardExpiry
     * @return RequestModel
     */
    public function setCreditCardExpiry($creditCardExpiry)
    {
        $this->creditCardExpiry = $creditCardExpiry;
        return $this;
    }

    /**
     * @return string
     */
    public function getCreditCardBrand()
    {
        return $this->creditCardBrand;
    }

    /**
     * @param string $creditCardBrand
     * @return RequestModel
     */
    public function setCreditCardBrand($creditCardBrand)
    {
        $this->creditCardBrand = $creditCardBrand;
        return $this;
    }

    /**
     * @return int|string
     */
    public function getCapture()
    {
        return $this->capture;
    }

    /**
     * @param int|string $capture
     * @return RequestModel
     */
    public function setCapture($capture)
    {
        $this->capture = $capture;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderDescription()
    {
        return $this->orderDescription;
    }

    /**
     * @param string $orderDescription
     * @return RequestModel
     */
    public function setOrderDescription($orderDescription)
    {
        $this->orderDescription = $orderDescription;
        return $this;
    }

    /**
     * @return string
     */
    public function getMessageVersion()
    {
        return $this->messageVersion;
    }

    /**
     * @param string $messageVersion
     * @return RequestModel
     */
    public function setMessageVersion($messageVersion)
    {
        $this->messageVersion = $messageVersion;
        return $this;
    }

    /**
     * @return CredentialOnFile
     */
    public function getCredentialOnFile()
    {
        return $this->credentialOnFile;
    }

    /**
     * @param CredentialOnFile $credentialOnFile
     * @return RequestModel
     */
    public function setCredentialOnFile($credentialOnFile)
    {
        $this->credentialOnFile = $credentialOnFile;
        return $this;
    }

    /**
     * @return Card
     */
    public function getCard()
    {
        return $this->card;
    }

    /**
     * @param Card $card
     * @return RequestModel
     */
    public function setCard($card)
    {
        $this->card = $card;
        return $this;
    }

    /**
     * @return string
     */
    public function getSchemeReferenceId()
    {
        return $this->schemeReferenceId;
    }

    /**
     * @param $schemeReferenceId
     * @return RequestModel
     */
    public function setSchemeReferenceId($schemeReferenceId)
    {
        $this->schemeReferenceId = $schemeReferenceId;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameterData()
    {
        $parameterData = parent::getParameterData();

        $parameterData['CCNr'] = $this->creditCardNumber;
        $parameterData['CCCVC'] = $this->creditCardValidationCode;
        $parameterData['CCExpiry'] = $this->creditCardExpiry;
        $parameterData['CCBrand'] = $this->creditCardBrand;
        $parameterData['Capture'] = $this->capture;
        $parameterData['OrderDesc'] = $this->orderDescription;
        $parameterData['MsgVer'] = $this->messageVersion;

        $parameterData['Card'] = !empty($this->card)
            ? $this->encode($this->card)
            : null;

        $parameterData['credentialOnFile'] = !empty($this->credentialOnFile)
            ? $this->encode($this->credentialOnFile)
            : null;

        $parameterData['schemeReferenceID'] = $this->schemeReferenceId;

        return $parameterData;
    }
}
