<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Utility;

use LifeStyle\ComputopPaygateAdapter\Exception\ArgumentTypeException;
use LifeStyle\ComputopPaygateAdapter\Model\ParameterProviderInterface;
use LifeStyle\ComputopPaygateAdapter\Model\RequestModel;
use LifeStyle\ComputopPaygateAdapter\Model\ResponseModel;

/**
 * Class ResponseFactory
 * @package LifeStyle\ComputopPaygateAdapter\Utility
 */
final class ResponseFactory
{
    /**
     * RequestModelConverter constructor.
     */
    private function __construct()
    {
    }

    /**
     * @param RequestModel $requestModel
     * @return ResponseModel
     * @throws ArgumentTypeException
     */
    public static function createResponseFromRequestModel($requestModel)
    {
        // The HMAC has to be calculated for verification.
        $mac = self::generateMessageAuthenticationCode($requestModel->getPaymentId(),
            $requestModel->getTransactionId(), $requestModel->getMerchantId(), $requestModel->getAmount(),
            $requestModel->getCurrency());
        $requestModel->setMac($mac);

        return self::createResponse($requestModel, $requestModel->getTarget(),
            $requestModel->getMerchantId());
    }

    /**
     * @param ParameterProviderInterface $parameterProvider
     * @param string $target
     * @param string $merchantId
     * @return ResponseModel
     * @throws ArgumentTypeException
     */
    public static function createResponse($parameterProvider, $target, $merchantId)
    {
        // Make sure the type is correct.

        if (!is_object($parameterProvider)) {
            $type = gettype($parameterProvider);
            throw new ArgumentTypeException("Expected an object of parameter provider, got '{$type}'!");
        }
        if (!($parameterProvider instanceof ParameterProviderInterface)) {
            $class = get_class($parameterProvider);
            throw new ArgumentTypeException("Expected an instance of parameter provider, got '{$class}'!");
        }

        $query = self::buildQuery($parameterProvider);

        $queryLength = strlen($query);
        $queryData = Security::encrypt($query);

        return (new ResponseModel())
            ->setTarget($target)
            ->setMerchantId($merchantId)
            ->setLength($queryLength)
            ->setData($queryData);
    }

    /**
     * Create a valid `GET` target.
     *
     * @param ResponseModel $responseModel
     * @return string
     * @deprecated Not supported. Use a `POST` instead.
     */
    public static function createTarget($responseModel)
    {
        // Make sure the type is correct.

        if (!is_object($responseModel)) {
            $type = gettype($responseModel);
            throw new ArgumentTypeException("Expected an object of response model, got '{$type}'!");
        }
        if (!($responseModel instanceof ResponseModel)) {
            $class = get_class($responseModel);
            throw new ArgumentTypeException("Expected an instance of response model, got '{$class}'!");
        }

        $targetBase = $responseModel->getTarget();
        $targetQuery = self::buildQuery($responseModel);

        return "{$targetBase}?{$targetQuery}";
    }

    /**
     * @param string $paymentId
     * @param string $transactionId
     * @param string $merchantId
     * @param int $amount
     * @param string $currency
     * @return string
     */
    public static function generateMessageAuthenticationCode($paymentId, $transactionId, $merchantId, $amount, $currency) {
        $code = "{$paymentId}*{$transactionId}*{$merchantId}*{$amount}*{$currency}";
        return Security::hash($code);
    }

    /**
     * @param ParameterProviderInterface $parameterProvider
     * @return string
     */
    private static function buildQuery($parameterProvider)
    {
        $parameterData = $parameterProvider->getParameterData();
        $parameterData = array_filter($parameterData);

        $query = http_build_query($parameterData, null, '&', PHP_QUERY_RFC1738);
        // The query must not be urlencoded.
        return urldecode($query);
    }
}
