<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo;

use JsonSerializable;

/**
 * Class threeDSData
 * @package LifeStyle\ComputopPaygateAdapter\Model\ThreeDSecureTwo
 */
class ThreeDSData implements JsonSerializable
{
    /**
     * `acsProtocolVersion` -- The protocol version used for authentication.
     *  Values accepted `['1.0.2', '2.1.0', '2.2.0', '2.3.0']`
     *
     * @var string
     */
    private $acsProtocolVersion;

    /**
     * `authenticationValue` -- Payment system specific value to provide proof of authentication
     *
     * @var string
     */
    private $authenticationValue;

    /**
     * `eci` -- Payment system specific Electronic Commerce Indicator
     *
     * @var string
     */
    private $eci;

    /**
     * `threeDSServerTransID` -- 3DS 2.0 only. Universally unique transaction identifier assigned by the 3DS Server.
     *
     * @var string
     */
    private $threeDSServerTransID;

    /**
     * `acsXID` -- 3DS 1.0 only. ACS assigned transaction ID.
     *
     * @var string
     */
    private $acsXID;

    /**
     * `dsTransID` -- 3DS 2.0 only. Universally unique transaction identifier assigned by the DS to identify a single transaction.
     *
     * @var string
     */
    private $dsTransID;

    /**
     * `intermediateStatus` -- This is the transaction status transStatus from the Authentication Response (ARes).
     *
     * Values accepted:
     * Y = Authentication Verification Successful.
     * N = Not Authenticated /Account Not Verified; Transaction denied.
     * U = Authentication/ Account Verification Could Not Be Performed; Technical or other problem, as indicated in ARes or RReq.
     * A = Attempts Processing Performed; Not Authenticated/Verified, but a proof of attempted authentication/verification is provided.
     * C = Challenge Required; Additional authentication is required using the CReq/CRes.
     * D = Challenge Required; Decoupled Authentication confirmed.
     * I = Informational Only; 3DS Requestor challenge preference acknowledged.
     *
     * @var string
     */
    private $intermediateStatus;

    /**
     * `finalStatus` -- This is the transaction status transStatus from the Result Request (RReq).
     *
     * Values accepted:
     * Y = Authentication Verification Successful.
     * U = Authentication / Account Verification Could Not Be Performed; Technical or other problem, as indicated in ARes or RReq.
     * A = Attempts Processing Performed; Not Authenticated/Verified, but a proof of attempted authentication/verification is provided.
     *
     * @var string
     */
    private $finalStatus;

    /**
     * @return string
     */
    public function getAcsProtocolVersion()
    {
        return $this->acsProtocolVersion;
    }

    /**
     * @param string $acsProtocolVersion
     * @return threeDSData
     */
    public function setAcsProtocolVersion($acsProtocolVersion)
    {
        $this->acsProtocolVersion = $acsProtocolVersion;
        return $this;
    }

    /**
     * @return string
     */
    public function getAuthenticationValue()
    {
        return $this->authenticationValue;
    }

    /**
     * @param string $authenticationValue
     * @return threeDSData
     */
    public function setAuthenticationValue($authenticationValue)
    {
        $this->authenticationValue = $authenticationValue;
        return $this;
    }

    /**
     * @return string
     */
    public function getEci()
    {
        return $this->eci;
    }

    /**
     * @param string $eci
     * @return threeDSData
     */
    public function setEci($eci)
    {
        $this->eci = $eci;
        return $this;
    }

    /**
     * @return string
     */
    public function getThreeDSServerTransID()
    {
        return $this->threeDSServerTransID;
    }

    /**
     * @param string $threeDSServerTransID
     * @return threeDSData
     */
    public function setThreeDSServerTransID($threeDSServerTransID)
    {
        $this->threeDSServerTransID = $threeDSServerTransID;
        return $this;
    }

    /**
     * @return string
     */
    public function getAcsXID()
    {
        return $this->acsXID;
    }

    /**
     * @param string $acsXID
     * @return threeDSData
     */
    public function setAcsXID($acsXID)
    {
        $this->acsXID = $acsXID;
        return $this;
    }

    /**
     * @return string
     */
    public function getDsTransID()
    {
        return $this->dsTransID;
    }

    /**
     * @param string $dsTransID
     * @return threeDSData
     */
    public function setDsTransID($dsTransID)
    {
        $this->dsTransID = $dsTransID;
        return $this;
    }

    /**
     * @return string
     */
    public function getIntermediateStatus()
    {
        return $this->intermediateStatus;
    }

    /**
     * @param string $intermediateStatus
     * @return threeDSData
     */
    public function setIntermediateStatus($intermediateStatus)
    {
        $this->intermediateStatus = $intermediateStatus;
        return $this;
    }

    /**
     * @return string
     */
    public function getFinalStatus()
    {
        return $this->finalStatus;
    }

    /**
     * @param string $finalStatus
     * @return threeDSData
     */
    public function setFinalStatus($finalStatus)
    {
        $this->finalStatus = $finalStatus;
        return $this;
    }

    /**
     * @return array
     * @internal
     */
    public function jsonSerialize()
    {
        return array_filter([
            'acsProtocolVersion' => $this->acsProtocolVersion,
            'authenticationValue' => $this->authenticationValue,
            'eci' => $this->eci,
            'threeDSServerTransID' => $this->threeDSServerTransID,
            'acsXID' => $this->acsXID,
            'dsTransID' => $this->dsTransID,
            'intermediateStatus' => $this->intermediateStatus,
            'finalStatus' => $this->finalStatus,
        ], function ($value) {
            // Only null values should be filtered out, keeping the false values.
            return $value !== null;
        });
    }
}
