<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter;

use LifeStyle\ComputopPaygateAdapter\Exception\ArgumentTypeException;
use LifeStyle\ComputopPaygateAdapter\Exception\PayTypeHandlerNotFoundException;
use LifeStyle\ComputopPaygateAdapter\Model\RequestModel;
use LifeStyle\ComputopPaygateAdapter\Model\ResponseModel;
use LifeStyle\ComputopPaygateAdapter\Paytype\PaytypeHandlerInterface;

/**
 * Class PaygateService
 * @package LifeStyle\ComputopPaygateAdapter
 */
class PaygateService
{
    /**
     * @var array|PaytypeHandlerInterface[]
     */
    private $paytypeHandlerList;

    /**
     * PaygateService constructor.
     */
    public function __construct()
    {
        $this->paytypeHandlerList = [];
    }

    /**
     * @param string $paytypeName
     * @param RequestModel $requestModel
     * @return ResponseModel
     * @throws ArgumentTypeException
     */
    public function create($paytypeName, $requestModel)
    {
        $paytypeHandler = $this->getPaytypeHandler($paytypeName);
        if (is_null($paytypeHandler)) {
            throw new PayTypeHandlerNotFoundException("No paytype handler found for given paytype name '{$paytypeName}'.");
        }

        return $paytypeHandler->handle($requestModel);
    }

    /**
     * @param PaytypeHandlerInterface $paytypeHandler
     * @return PaygateService
     * @throws ArgumentTypeException
     */
    public function addPaytypeHandler($paytypeHandler)
    {
        // Make sure the type is correct.

        if (!is_object($paytypeHandler)) {
            $type = gettype($paytypeHandler);
            throw new ArgumentTypeException("Expected an object of paytype handler, got '{$type}'!");
        }
        if (!($paytypeHandler instanceof PaytypeHandlerInterface)) {
            $class = get_class($paytypeHandler);
            throw new ArgumentTypeException("Expected an instance of paytype handler, got '{$class}'!");
        }

        $this->paytypeHandlerList[] = $paytypeHandler;
        return $this;
    }

    /**
     * @param string $paytypeName
     * @return PaytypeHandlerInterface|null
     */
    protected function getPaytypeHandler($paytypeName)
    {
        foreach ($this->paytypeHandlerList as $paytypeHandler) {
            if (!$paytypeHandler->canHandle($paytypeName)) {
                continue;
            }

            return $paytypeHandler;
        }

        return null;
    }
}
