<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ComputopPaygateAdapter\Utility;

use LifeStyle\ComputopPaygateAdapter\Exception\ArgumentTypeException;
use LifeStyle\ComputopPaygateAdapter\Model\ParameterProviderInterface;
use LifeStyle\ComputopPaygateAdapter\Model\RequestModel;
use LifeStyle\ComputopPaygateAdapter\Model\ResponseModel;

/**
 * Class Secure
 * @package LifeStyle\ComputopPaygateAdapter\Utility
 */
final class ResponseFactory
{
    /**
     * @var string|null
     */
    private static $KEY = null;

    /**
     * @var string|null
     */
    private static $KEY_HASH = null;

    /**
     * @param string|null $key
     */
    public static function setKey($key)
    {
        self::$KEY = $key;
    }

    /**
     * @param string|null $keyHmac
     */
    public static function setKeyHash($keyHmac)
    {
        self::$KEY_HASH = $keyHmac;
    }

    /**
     * RequestModelConverter constructor.
     */
    private function __construct()
    {
    }

    /**
     * @param RequestModel $requestModel
     * @return ResponseModel
     * @throws ArgumentTypeException
     */
    public static function createResponseFromRequestModel($requestModel)
    {
        // The HMAC has to be calculated for verification.
        $mac = ResponseFactory::generateMessageAuthenticationCode($requestModel->getPaymentId(),
            $requestModel->getTransactionId(), $requestModel->getMerchantId(), $requestModel->getAmount(),
            $requestModel->getCurrency());
        $requestModel->setMac($mac);

        return static::createResponse($requestModel, $requestModel->getTarget(),
            $requestModel->getMerchantId());
    }

    /**
     * @param ParameterProviderInterface $parameterProvider
     * @param string $target
     * @param string $merchantId
     * @return ResponseModel
     * @throws ArgumentTypeException
     */
    public static function createResponse($parameterProvider, $target, $merchantId)
    {
        // Make sure the type is correct.

        if (!is_object($parameterProvider)) {
            $type = gettype($parameterProvider);
            throw new ArgumentTypeException("Expected an object of parameter provider, got '{$type}'!");
        }
        if (!is_subclass_of($parameterProvider, ParameterProviderInterface::class, false)) {
            $class = get_class($parameterProvider);
            throw new ArgumentTypeException("Expected an instance of parameter provider, got '{$class}'!");
        }

        $query = self::buildQuery($parameterProvider);

        $queryLength = strlen($query);
        $queryData = self::encrypt($query);

        return (new ResponseModel())
            ->setTarget($target)
            ->setMerchantId($merchantId)
            ->setLength($queryLength)
            ->setData($queryData);
    }

    /**
     * Create a valid `GET` target.
     *
     * @param ResponseModel $responseModel
     * @return string
     */
    public static function createTarget($responseModel)
    {
        $targetBase = $responseModel->getTarget();
        $targetQuery = self::buildQuery($responseModel);

        return "{$targetBase}?{$targetQuery}";
    }

    /**
     * @param string $paymentId
     * @param string $transactionId
     * @param string $merchantId
     * @param int $amount
     * @param string $currency
     * @return string
     */
    public static function generateMessageAuthenticationCode($paymentId, $transactionId, $merchantId, $amount, $currency)
    {
        $code = "{$paymentId}*{$transactionId}*{$merchantId}*{$amount}*{$currency}";
        return self::hash($code);
    }

    /**
     * @param ParameterProviderInterface $parameterProvider
     * @return string
     */
    private static function buildQuery($parameterProvider)
    {
        $parameterData = $parameterProvider->getParameterData();
        $parameterData = array_filter($parameterData);

        $query = http_build_query($parameterData);
        // The query must not be urlencoded.
        return urldecode($query);
    }

    /**
     * @param string $input
     * @return string
     */
    private static function hash($input)
    {
        $output = hash_hmac('sha256', $input, self::$KEY_HASH, true);
        return bin2hex($output);
    }

    /**
     * See the {@see https://www.php.net/manual/en/function.openssl-encrypt.php#121545 source}.
     *
     * @param string $input
     * @return string
     */
    private static function encrypt($input)
    {
        // Fill the input with null-byte character.
        if ($length = strlen($input) % 8) {
            $input = $input . str_repeat(0x00, 8 - $length);
        }

        $output = '';
        if (function_exists('mcrypt_encrypt')) {
            $output = mcrypt_encrypt(MCRYPT_BLOWFISH, self::$KEY, $input, MCRYPT_MODE_ECB, null);
        } else {
            $output = openssl_encrypt($input, 'BF-ECB', self::$KEY, OPENSSL_RAW_DATA | OPENSSL_NO_PADDING);
            if (!is_string($output)) {
                $output = '';
            }
        }

        return bin2hex($output);
    }
}
