<?php

/**
 * Class DataCollectorTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\DataCollector;

use JMS\Serializer\SerializerBuilder;
use PHPUnit\Framework\TestCase;

/**
 * Class DataCollectorTest
 * @package Lifestyle\DataCollector
 */
class DataCollectorTest extends TestCase
{

    /**
     * @var string
     */
    public static $timezone;

    /**
     * set timezone
     */
    public static function setUpBeforeClass()
    {
        static::$timezone = date_default_timezone_get();
        date_default_timezone_set('Europe/Berlin');
    }

    /**
     * cleanup after tests
     */
    public static function tearDownAfterClass()
    {
        date_default_timezone_set(static::$timezone);
    }

    public function testAddStandardItem()
    {
        $itemKey = 'standard_item';
        $itemValue = 'standard_item_value';

        $collector = new DataCollector();
        $collector->addItem($itemKey, $itemValue);

        $this->assertTrue($collector->hasItem($itemKey));
        $this->assertInstanceOf(DataCollectorItem::class, $collector->getItem($itemKey));
        $this->assertInstanceOf(DataCollectorItemInterface::class, $collector->getItem($itemKey));
        $this->assertSame($itemValue, $collector->getItemValue($itemKey));
        $this->assertFalse($collector->getItem($itemKey)->hasLocale());
        $this->assertCount(1, $collector->getItems());
    }

    public function testAddTranslatedItem()
    {
        $itemKey = 'translated_item';
        $itemValueDE = 'translated_item_DE';
        $itemValueEN = 'translated_item_EN';

        $localeDE = 'de_DE';
        $localeEN = 'en_EN';

        $collector = new DataCollector();
        $collector->addItem($itemKey, $itemValueDE, $localeDE);
        $collector->addItem($itemKey, $itemValueEN, $localeEN);

        // Assertions for first $localeDE
        $this->assertTrue($collector->hasItem($itemKey, $localeDE));
        $this->assertInstanceOf(DataCollectorItem::class, $collector->getItem($itemKey, $localeDE));
        $this->assertInstanceOf(DataCollectorItemInterface::class, $collector->getItem($itemKey, $localeDE));
        $this->assertSame($itemValueDE, $collector->getItemValue($itemKey, $localeDE));
        $this->assertTrue($collector->getItem($itemKey, $localeDE)->hasLocale());
        $this->assertCount(2, $collector->getItems());
    }

    public function testResetTranslatedItem()
    {
        $itemKey = 'standard_item';
        $itemValue = 'standard_item_value';

        $collector = new DataCollector();
        $collector->addItem($itemKey, $itemValue);

        $this->assertTrue($collector->hasItem($itemKey));
        $this->assertInstanceOf(DataCollectorItem::class, $collector->getItem($itemKey));
        $this->assertInstanceOf(DataCollectorItemInterface::class, $collector->getItem($itemKey));
        $this->assertSame($itemValue, $collector->getItemValue($itemKey));
        $this->assertFalse($collector->getItem($itemKey)->hasLocale());
        $this->assertCount(1, $collector->getItems());

        $collectorItem = $collector->getItem($itemKey);
        $collectorItem->setValue('new_value');
        $collectorItem->setLocale('new_locale');

        $this->assertFalse($collector->hasItem($itemKey));
        $this->assertTrue($collector->hasItem($itemKey, 'new_locale'));
        $this->assertSame('new_value', $collector->getItemValue($itemKey, 'new_locale'));
        $this->assertSame('new_locale', $collector->getItem($itemKey, 'new_locale')->getLocale());
    }

    public function testGetItemWillReturnNull()
    {
        $collector = new DataCollector();
        $collector->addItem('some_item_key', 'some_item_value');

        $this->assertNull($collector->getItem('not_existing_item_key'));
    }

    public function testGetItemValueWillReturnNull()
    {
        $collector = new DataCollector();
        $collector->addItem('some_item_key', 'some_item_value');

        $this->assertNull($collector->getItemValue('not_existing_item_key'));
    }

    public function testGetItemsByLocale()
    {
        $localeDE = 'de_DE';
        $localeEN = 'en_EN';

        $collector = new DataCollector();
        $collector->addItem('de_DE_key_1', 'de_DE_value_1', $localeDE);
        $collector->addItem('de_DE_key_2', 'de_DE_value_2', $localeDE);
        $collector->addItem('de_DE_key_3', 'de_DE_value_3', $localeDE);

        $collector->addItem('en_EN_key_1', 'en_EN_value_1', $localeEN);
        $collector->addItem('en_EN_key_2', 'en_EN_value_2', $localeEN);
        $collector->addItem('en_EN_key_3', 'en_EN_value_3', $localeEN);

        $this->assertCount(3, $collector->getItems($localeDE));
        $this->assertCount(3, $collector->getItems($localeEN));
        $this->assertSame([], $collector->getItems('unknown_LOCALE'));
    }

    public function testGetCollectedLocales()
    {
        $collector = new DataCollector();
        $collector->addItem('de_DE_key', 'de_DE_value', 'de_DE');
        $collector->addItem('en_DE_key', 'en_DE_value', 'en_DE');
        $collector->addItem('en_US_key', 'en_US_value', 'en_US');
        $collector->addItem('en_EN_key', 'en_EN_value', 'en_EN');

        $collectedLocales = $collector->getCollectedLocales();

        $this->assertCount(4, $collectedLocales);
        $this->assertSame('de_DE', $collectedLocales[0]);
        $this->assertSame('en_DE', $collectedLocales[1]);
        $this->assertSame('en_US', $collectedLocales[2]);
        $this->assertSame('en_EN', $collectedLocales[3]);
    }

    public function testAddExistingItemWillDoNothing()
    {
        // Test without locale
        $collector = new DataCollector();
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value');
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value');

        $this->assertCount(1, $collector->getItems());

        // Test with locale
        $collector = new DataCollector();
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value', 'en_US');
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value', 'en_US');

        $this->assertCount(1, $collector->getItems());
    }

    public function testReplaceItemWithoutLocale()
    {
        $itemKey = 'unit_test_unique_key';

        $collector = new DataCollector();
        $collector->addItem($itemKey, 'unit_test_unique_value');

        $item = new DataCollectorItem($itemKey, 'unit_test_replaced_value');
        $collector->replaceItem($item);

        $this->assertCount(1, $collector->getItems());
        $this->assertTrue($collector->hasItem($itemKey));
        $this->assertSame('unit_test_replaced_value', $collector->getItem($itemKey)->getValue());
    }

    public function testReplaceItemWithLocale()
    {
        $itemKey = 'unit_test_unique_key';
        $locale = 'unit_TEST';

        $collector = new DataCollector();
        $collector->addItem($itemKey, 'unit_test_unique_value', $locale);

        $item = new DataCollectorItem($itemKey, 'unit_test_replaced_value', $locale);
        $collector->replaceItem($item);

        $this->assertCount(1, $collector->getItems());
        $this->assertTrue($collector->hasItem($itemKey, $locale));
        $this->assertSame('unit_test_replaced_value', $collector->getItem($itemKey, $locale)->getValue());
    }

    public function testReplaceItemWillAddNewItemWithoutLocale()
    {
        $collector = new DataCollector();
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value');

        $item = new DataCollectorItem('new_item', 'some_value');
        $collector->replaceItem($item);

        $this->assertCount(2, $collector->getItems());
        $this->assertTrue($collector->hasItem('unit_test_unique_key'));
        $this->assertSame('unit_test_unique_value', $collector->getItem('unit_test_unique_key')->getValue());

        $this->assertTrue($collector->hasItem('new_item'));
        $this->assertSame('some_value', $collector->getItem('new_item')->getValue());
    }

    public function testReplaceItemWillAddNewItemWithLocale()
    {
        $locale = 'some_LOCALE';

        $collector = new DataCollector();
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value', $locale);

        $item = new DataCollectorItem('new_item', 'some_value', $locale);
        $collector->replaceItem($item);

        $this->assertCount(2, $collector->getItems());
        $this->assertTrue($collector->hasItem('unit_test_unique_key', $locale));
        $this->assertSame('unit_test_unique_value', $collector->getItem('unit_test_unique_key', $locale)->getValue());

        $this->assertTrue($collector->hasItem('new_item', $locale));
        $this->assertSame('some_value', $collector->getItem('new_item', $locale)->getValue());
    }

    public function testSerializeJson()
    {
        $collector = $this->getDataCollectorForSerializationTest();

        $serializer = SerializerBuilder::create()->build();

        $this->assertJsonStringEqualsJsonFile(
            __DIR__ . '/Serializer/data_collector_serialized.json',
            $serializer->serialize($collector, 'json')
        );
    }

    public function testDeserializeJson()
    {
        $serializer = SerializerBuilder::create()->build();

        /** @var DataCollectorInterface $collector */
        $collector = $serializer->deserialize(
            file_get_contents(__DIR__ . '/Serializer/data_collector_serialized.json'), DataCollector::class, 'json'
        );

        $expectedCollector = $this->getDataCollectorForSerializationTest();

        $this->assertSame($expectedCollector->getItemValue('de_DE_key'), $collector->getItemValue('de_DE_key'));
        $this->assertSame($expectedCollector->getItemValue('en_DE_key'), $collector->getItemValue('en_DE_key'));
        $this->assertSame($expectedCollector->getItemValue('en_US_key'), $collector->getItemValue('en_US_key'));
        $this->assertSame($expectedCollector->getItemValue('en_EN_key'), $collector->getItemValue('en_EN_key'));
        $this->assertSame($expectedCollector->getItemValue('unit_test_unique_key'), $collector->getItemValue('unit_test_unique_key'));
        $this->assertSame($expectedCollector->getItemValue('integer_test'), $collector->getItemValue('integer_test'));
        $this->assertSame($expectedCollector->getItemValue('float_test'), $collector->getItemValue('float_test'));
        $this->assertSame($expectedCollector->getItemValue('array_test'), $collector->getItemValue('array_test'));

        $this->assertInstanceOf(\DateTime::class, $collector->getItemValue('object_test'));

        /** @var \DateTime $expectedDateTime */
        $expectedDateTime = $expectedCollector->getItemValue('object_test');
        $expectedDateTime->getTimezone()->getName();
        $expectedDateTime->getTimestamp();

        /** @var \DateTime $actualDateTime */
        $actualDateTime = $collector->getItemValue('object_test');
        $actualDateTime->getTimezone()->getName();
        $actualDateTime->getTimestamp();

        $this->assertSame($expectedDateTime->getTimezone()->getName(), $actualDateTime->getTimezone()->getName());
        $this->assertSame($expectedDateTime->getTimestamp(), $actualDateTime->getTimestamp());
    }

    public function testRemoveStandardItem()
    {
        $toBeRemovedKey = 'to_be_removed';

        $collector = new DataCollector();
        $collector->addItem($toBeRemovedKey, 'remove_this_item');
        $collector->addItem($toBeRemovedKey, 'do_not_remove_this_one', 'test_LOCALE');

        $collector->removeItem($toBeRemovedKey);

        $this->assertFalse($collector->hasItem($toBeRemovedKey));
        $this->assertTrue($collector->hasItem($toBeRemovedKey, 'test_LOCALE'));
    }

    public function testRemoveLocalizedItem()
    {
        $toBeRemovedKey = 'to_be_removed';

        $collector = new DataCollector();
        $collector->addItem($toBeRemovedKey, 'do_not_remove_this_one');
        $collector->addItem($toBeRemovedKey, 'remove_this_item', 'test_LOCALE');

        $collector->removeItem($toBeRemovedKey, 'test_LOCALE');

        $this->assertFalse($collector->hasItem($toBeRemovedKey, 'test_LOCALE'));
        $this->assertTrue($collector->hasItem($toBeRemovedKey));
    }

    private function getDataCollectorForSerializationTest()
    {
        $collector = new DataCollector();
        $collector->addItem('de_DE_key', 'de_DE_value', 'de_DE');
        $collector->addItem('en_DE_key', 'en_DE_value', 'en_DE');
        $collector->addItem('en_US_key', 'en_US_value', 'en_US');
        $collector->addItem('en_EN_key', 'en_EN_value', 'en_EN');
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value');
        $collector->addItem('integer_test', 123);
        $collector->addItem('float_test', 123.123);
        $collector->addItem('array_test', ['1' => ['test'], 3, ['test2' => 'lorem ipsum']]);
        $collector->addItem(
            'object_test',
            \DateTime::createFromFormat('d-M-Y H:i:s', '12-Dec-2017 12:12:12')
        );

        return $collector;
    }
}
