# generic-sitemap

This package provides a way to generate xml sitemaps.

It aims to be reusable and very generic, yet provides functionality to generate sitemaps that meet the [google sitemap guidelines and requirements](https://support.google.com/webmasters/answer/183668?hl=en#sitemapformat) out of the box.


## Installation

To install the package, simply add it to the _repositories_ section in _composer.json_ of your existing project.

```json
{
  "type": "vcs",
  "url": "ssh://git@stash.life-style.de:7999/lcp/generic-sitemap.git"
}
```

After that, you can require it via composer:

```bash
composer require lifestyle/generic-sitemap
```


## Usage

For details read the [documentation section](#documentation).

There is a general usage example given inside [sample.php](sample.php).

Some additional documentation is available as phpdoc comments.


## Documentation

Some information about the most important features:

**1. Generator**:

Each _SitemapGenerator_ instance should be created using the _SitemapGeneratorBuilder_:

```php
$builder = \GenericSitemap\Builder\SitemapGeneratorBuilder::create();

$generator = $builder
  ->build();
```

* Available configuration options are:
    
  * _SitemapSerializer_: Default is _JMSSitemapSerializer_
  
  ```php
  $builder->setSitemapSerializer($serializer);
  ```
  
  When you don't want to use JMS for serialization, you can implement your own serializer using the _SitemapSerializerInterface_.
  
  In case you add the _SitemapGeneratorType_ or _SitemapIndexGeneratorType_ by hand and don't let the default handle that, this option will have no effect!
  
  * _Formatter_: Default is _SitemapUrlFormatter_
  
  ```php
  $builder->setUrlFormatter($formatter);
  ```
  
  Or use default with custom url (recommended):
  
  ```php
  $builder->setUrlFormatterForUrl($url);
  ```
  
  Setting this option is necessary, as it is used to generate the urls for the sitemap index.
  
  In case you add the _SitemapIndexGeneratorType_ by hand and don't let the default handle that, this option will have no effect!
  
  * _SitemapGeneratorType_: Default is _[SitemapGeneratorType, SitemapIndexGeneratorType]_
    
  ```php
  $builder->setGeneratorTypeList($generatorTypeList);
  ```
  
  Or progressively add generator types to the list:
  
  ```php
  $builder
      ->addGeneratorType($generatorType1)
      ->addGeneratorType($generatorType2);
  ```
  
  For most use cases the default should be sufficient.
  
  * _$maxUrlsetSize_: Default is _DEFAULT_MAX_URLSET_SIZE = 5000_
  
  ```php
  $builder->setMaxUrlsetSize($maxUrlsetSize);
  ```
  
  Defines the maximum size of an urlset per sitemap and thus sets, when a sitemap has to be split.
  
  In case you set the _SitemapGeneratorInputNormalizer_ by hand and don't let the default handle that, this option will have no effect!
  
  * _SitemapGeneratorInputNormalizer_: Default is _SitemapGeneratorInputNormalizer_
  
  ```php
  $builder->setSitemapGeneratorInputNormalizer($sitemapGeneratorInputNormalizer);
  ```
  
  If the _SitemapGeneratorInput_ needs another set of normalizations, you can implement your own using the _SitemapGeneratorInputNormalizerInterface_.


**2. Model classes**:

The _SitemapGenerator_ has just one method, _generate()_, which takes an _SitemapGeneratorInput_ object as argument.

Example on how to build such an object (there are multiple possible ways):

```php
$input = (new \GenericSitemap\Model\SitemapGeneratorInput())
    ->setSitemapInputList([
        (new \GenericSitemap\Model\SitemapInput('some-name'))
            // Set urlset list
            ->setUrlsetList([
                (new \GenericSitemap\Model\Sitemap\Urlset())
                    // ...
            ])
            // Add urlset
            ->addUrlset(
                (new \GenericSitemap\Model\Sitemap\Urlset())
                    ->setUrlset([
                        // Set urls
                    ])
                    // Add url
                    ->addUrl(
                        (new \GenericSitemap\Model\Sitemap\Url(''))
                            // Lastmod = 7 days ago
                            ->setLastmod(
                                date('Y-m-d', time() - (((60 * 60) * 24) * 7))
                            )
                            // Changefreq = weekly
                            ->setChangefreq(
                                \GenericSitemap\Model\Sitemap\ChangeFreq::WEEKLY
                            )
                            // Priority = 0.5 aka default of search engine if not given
                            ->setPriority(0.5)
                    )
            )
    ])
    // Add sitemapinput with one urlset
    ->addData('some-other-name', (new \GenericSitemap\Model\Sitemap\Urlset())
        // ...
    );
```

All model classes support fluent setters for simplified inline instance creation.

Some also contain _addData()_ which is a shorthand to calling _addXYZ()_ with an instance of _XYZ_ as argument.


## Troubleshooting

For troubleshooting you can simply use the _DebugGeneratorType_ and add it like this:

```php
$builder->addSitemapGeneratorType(
    new \GenericSitemap\Generator\Type\DebugGeneratorType($logger)
);
```

**Caution**: Make sure the _DebugGeneratorType_ is always the last generator type added, otherwise the logged output will be incomplete.

Example of how to make sure that is the case:

```php
function sortDebug($a, $b)
{
    if ($a instanceof \GenericSitemap\Generator\Type\DebugGeneratorType) {
        return 1;
    }
    
    return 0;
}

$generatorTypeList = $builder->getGeneratorTypeList();

if (usort($generatorTypeList, 'sortDebug')) {
    $builder->setGeneratorTypeList($generatorTypeList);
}
```

The above should be executed before calling _build()_ on _SitemapGeneratorBuilder_.

## Logging

As this is only a package, the implementing project is responsible for logging.


## Tests

The tests are located inside the _tests_ directory. Use phpunit to execute tests.

---

Thats it!
