<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2018 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace GenericSitemap\Utils;

use GenericSitemap\Model\Sitemap\ChangeFreq;
use GenericSitemap\Model\Sitemap\Lastmod;
use GenericSitemap\Model\Sitemap\Priority;
use GenericSitemap\Model\Sitemap\Urlset;
use GenericSitemap\Model\Sitemap\Url;
use GenericSitemap\Model\SitemapGeneratorInput;
use GenericSitemap\Model\SitemapInput;

/**
 * Class SitemapGeneratorInputNormalizer
 * @package GenericSitemap\Utils
 */
class SitemapGeneratorInputNormalizer implements SitemapGeneratorInputNormalizerInterface
{
    /**
     * @var int
     */
    private $maxUrlsetSize;

    /**
     * @var Escaper
     */
    private $escaper;

    /**
     * SitemapGeneratorInputNormalizer constructor.
     * @param int $maxUrlsetSize
     * @param Escaper $escaper
     */
    public function __construct($maxUrlsetSize, $escaper)
    {
        $this->maxUrlsetSize = $maxUrlsetSize;
        $this->escaper = $escaper;
    }

    /**
     * @param SitemapGeneratorInput $sitemapGeneratorInput
     * @return SitemapGeneratorInput
     */
    public function normalize($sitemapGeneratorInput)
    {
        return $this->normalizeSitemapGeneratorInput($sitemapGeneratorInput);
    }

    /**
     * @param SitemapGeneratorInput $sitemapGeneratorInput
     * @return SitemapGeneratorInput
     */
    private function normalizeSitemapGeneratorInput($sitemapGeneratorInput)
    {
        $normalizedSitemapGeneratorInput = new SitemapGeneratorInput();

        foreach ($sitemapGeneratorInput->getSitemapInputList() as $sitemapInput) {
            $normalizedSitemapGeneratorInput->addSitemapInput(
                $this->normalizeSitemapInput($sitemapInput)
            );
        }

        return $normalizedSitemapGeneratorInput;
    }

    /**
     * @param SitemapInput $sitemapInput
     * @return SitemapInput
     */
    private function normalizeSitemapInput($sitemapInput)
    {
        $name = $this->normalizeName($sitemapInput->getName());

        $normalizedSitemapInput = new SitemapInput($name);

        foreach ($sitemapInput->getUrlsetList() as $urlset) {
            foreach ($this->splitUrlset($urlset) as $splitUrlset) {
                $normalizedSitemapInput->addUrlset(
                    $this->normalizeUrlset($splitUrlset)
                );
            }
        }

        return $normalizedSitemapInput;
    }

    /**
     * @param string $name
     * @return string
     */
    private function normalizeName($name)
    {
        return trim($name);
    }

    /**
     * Split urlset based on $maxUrlsetSize.
     *
     * @param Urlset $urlset
     * @return array|Urlset[]
     */
    private function splitUrlset($urlset)
    {
        return array_map(function ($chunkedUrlset) {
            $splitUrlset = new Urlset();
            $splitUrlset->setUrlset($chunkedUrlset);

            return $splitUrlset;
        }, array_chunk($urlset->getUrlset(), $this->maxUrlsetSize, false));
    }

    /**
     * @param Urlset $urlset
     * @return Urlset
     */
    private function normalizeUrlset($urlset)
    {
        $normalizedUrlset = new Urlset();

        foreach ($urlset->getUrlset() as $url) {
            $normalizedUrlset->addUrl(
                $this->normalizeUrl($url)
            );
        }

        return $normalizedUrlset;
    }

    /**
     * @param Url $url
     * @return Url
     */
    private function normalizeUrl($url)
    {
        $loc = $this->normalizeLoc($url->getLoc());
        $lastmod = $this->normalizeLastmod($url->getLastmod());
        $changefreq = $this->normalizeChangefreq($url->getChangefreq());
        $priority = $this->normalizePriority($url->getPriority());

        $url = new Url($loc);
        $url->setLastmod($lastmod);
        $url->setChangefreq($changefreq);
        $url->setPriority($priority);

        return $url;
    }

    /**
     * @param string $loc
     * @return string
     */
    private function normalizeLoc($loc)
    {
        return $this->escaper->escape($loc, false);
    }

    /**
     * @param null|string $lastmod
     * @return null|string
     */
    private function normalizeLastmod($lastmod)
    {
        if ($lastmod !== null) {
            $lastmod = trim($lastmod);

            if (Lastmod::isValid($lastmod)) {
                return $lastmod;
            }
        }

        return null;
    }

    /**
     * @param null|string $changefreq
     * @return null|string
     */
    private function normalizeChangefreq($changefreq)
    {
        if ($changefreq !== null) {
            $changefreq = trim($changefreq);

            if (ChangeFreq::isValid($changefreq)) {
                return $changefreq;
            }
        }

        return null;
    }

    /**
     * @param null|float $priority
     * @return null|float
     */
    private function normalizePriority($priority)
    {
        if ($priority !== null) {
            return Priority::getPriority($priority);
        }

        return null;
    }
}
