<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2018 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace GenericSitemap\Builder;

use GenericSitemap\Generator\SitemapGenerator;
use GenericSitemap\Generator\Type\SitemapGeneratorTypeInterface;
use GenericSitemap\Generator\Type\SitemapIndexGeneratorType;
use GenericSitemap\Generator\Type\SitemapGeneratorType;
use GenericSitemap\Serializer\JMSSitemapSerializer;
use GenericSitemap\Serializer\SitemapSerializerInterface;
use GenericSitemap\Utils\Escaper;
use GenericSitemap\Utils\FormatterInterface;
use GenericSitemap\Utils\SitemapGeneratorInputNormalizer;
use GenericSitemap\Utils\SitemapGeneratorInputNormalizerInterface;
use GenericSitemap\Utils\SitemapUrlFormatter;

/**
 * Class SitemapGeneratorBuilder
 * @package GenericSitemap\Builder
 */
final class SitemapGeneratorBuilder
{
    /**
     * @var int
     */
    const DEFAULT_MAX_URLSET_SIZE = 5000;

    /**
     * @var Escaper
     */
    private $escaper;

    /**
     * @var null|SitemapSerializerInterface
     */
    private $sitemapSerializer;

    /**
     * @var null|FormatterInterface
     */
    private $urlFormatter;

    /**
     * @var array|SitemapGeneratorTypeInterface[]
     */
    private $sitemapGeneratorTypeList;

    /**
     * @var int
     */
    private $maxUrlsetSize;

    /**
     * @var null|SitemapGeneratorInputNormalizerInterface
     */
    private $sitemapGeneratorInputNormalizer;

    /**
     * SitemapGeneratorBuilder constructor.
     */
    private function __construct()
    {
        $this->sitemapGeneratorTypeList = [];
        $this->maxUrlsetSize = 0;
    }

    public static function create()
    {
        return new static();
    }

    /**
     * @return SitemapGenerator
     */
    public function build()
    {
        if (!$this->escaper) {
            $this->setDefaultEscaper();
        }

        if (!$this->sitemapSerializer) {
            $this->setDefaultSerializer();
        }

        if (!$this->urlFormatter) {
            $this->setDefaultUrlFormatter();
        }

        if (!count($this->sitemapGeneratorTypeList)) {
            $this->setDefaultGeneratorTypeList();
        }

        if ($this->maxUrlsetSize <= 0) {
            $this->setDefaultMaxUrlsetSize();
        }

        if (!$this->sitemapGeneratorInputNormalizer) {
            $this->setDefaultSitemapGeneratorInputNormalizer();
        }

        return new SitemapGenerator(
            $this->sitemapGeneratorTypeList,
            $this->sitemapGeneratorInputNormalizer
        );
    }

    /**
     * @return Escaper
     */
    public function getEscaper()
    {
        return $this->escaper;
    }

    /**
     * @param Escaper $escaper
     * @return SitemapGeneratorBuilder
     */
    public function setEscaper($escaper)
    {
        $this->escaper = $escaper;
        return $this;
    }

    private function setDefaultEscaper()
    {
        return $this->setEscaper(
            new Escaper()
        );
    }

    /**
     * @return null|SitemapSerializerInterface
     */
    public function getSitemapSerializer()
    {
        return $this->sitemapSerializer;
    }

    /**
     * @param null|SitemapSerializerInterface $sitemapSerializer
     * @return SitemapGeneratorBuilder
     */
    public function setSitemapSerializer($sitemapSerializer)
    {
        $this->sitemapSerializer = $sitemapSerializer;
        return $this;
    }

    /**
     * @return SitemapGeneratorBuilder
     */
    private function setDefaultSerializer()
    {
        return $this->setSitemapSerializer(
            new JMSSitemapSerializer()
        );
    }

    /**
     * @return null|FormatterInterface
     */
    public function getUrlFormatter()
    {
        return $this->urlFormatter;
    }

    /**
     * @param null|FormatterInterface $urlFormatter
     * @return SitemapGeneratorBuilder
     */
    public function setUrlFormatter($urlFormatter)
    {
        $this->urlFormatter = $urlFormatter;
        return $this;
    }

    /**
     * @param string $baseUrl
     * @return SitemapGeneratorBuilder
     */
    public function setUrlFormatterForUrl($baseUrl)
    {
        $baseUrl = rtrim($baseUrl, '/');
        $this->urlFormatter = new SitemapUrlFormatter($baseUrl);
        return $this;
    }

    /**
     * @return SitemapGeneratorBuilder
     */
    private function setDefaultUrlFormatter()
    {
        return $this->setUrlFormatterForUrl('');
    }

    /**
     * @return array|SitemapGeneratorTypeInterface[]
     */
    public function getSitemapGeneratorTypeList()
    {
        return $this->sitemapGeneratorTypeList;
    }

    /**
     * @param array|SitemapGeneratorTypeInterface[] $sitemapGeneratorTypeList
     * @return SitemapGeneratorBuilder
     */
    public function setSitemapGeneratorTypeList($sitemapGeneratorTypeList)
    {
        $this->sitemapGeneratorTypeList = $sitemapGeneratorTypeList;
        return $this;
    }

    /**
     * @return SitemapGeneratorBuilder
     */
    private function setDefaultGeneratorTypeList()
    {
        return $this->setSitemapGeneratorTypeList([
            new SitemapGeneratorType($this->sitemapSerializer),
            new SitemapIndexGeneratorType($this->sitemapSerializer, $this->urlFormatter, $this->escaper),
        ]);
    }

    /**
     * @param SitemapGeneratorTypeInterface $sitemapGeneratorType
     * @return SitemapGeneratorBuilder
     */
    public function addSitemapGeneratorType($sitemapGeneratorType)
    {
        $this->sitemapGeneratorTypeList[] = $sitemapGeneratorType;
        return $this;
    }

    /**
     * @return int
     */
    public function getMaxUrlsetSize()
    {
        return $this->maxUrlsetSize;
    }

    /**
     * @param int $maxUrlsetSize
     * @return SitemapGeneratorBuilder
     */
    public function setMaxUrlsetSize($maxUrlsetSize)
    {
        $this->maxUrlsetSize = $maxUrlsetSize;
        return $this;
    }

    /**
     * @return SitemapGeneratorBuilder
     */
    private function setDefaultMaxUrlsetSize()
    {
        return $this->setMaxUrlsetSize(self::DEFAULT_MAX_URLSET_SIZE);
    }

    /**
     * @return null|SitemapGeneratorInputNormalizerInterface
     */
    public function getSitemapGeneratorInputNormalizer()
    {
        return $this->sitemapGeneratorInputNormalizer;
    }

    /**
     * @param null|SitemapGeneratorInputNormalizerInterface $sitemapGeneratorInputNormalizer
     * @return SitemapGeneratorBuilder
     */
    public function setSitemapGeneratorInputNormalizer($sitemapGeneratorInputNormalizer)
    {
        $this->sitemapGeneratorInputNormalizer = $sitemapGeneratorInputNormalizer;
        return $this;
    }

    /**
     * @return SitemapGeneratorBuilder
     */
    private function setDefaultSitemapGeneratorInputNormalizer()
    {
        return $this->setSitemapGeneratorInputNormalizer(
            new SitemapGeneratorInputNormalizer($this->maxUrlsetSize, $this->escaper)
        );
    }
}
