<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseCreditCard\Request\Version1;

use Lifestyle\PaymentServiceSdk\Exceptions\InvalidRequestParametersException;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseCreditCard\Request\RequestDataInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class RequestBuilder
 * @package Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseCreditCard\Request\Version1
 */
class RequestBuilder
{
    /**
     * @var Factory
     */
    private $factory;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * RequestBuilder constructor.
     * @param Factory $factory
     * @param ValidatorInterface $validator
     */
    public function __construct(Factory $factory, ValidatorInterface $validator)
    {
        $this->factory = $factory;
        $this->validator = $validator;
    }

    /**
     * @param RequestDataInterface $requestData
     * @return Request
     * @throws InvalidRequestParametersException
     */
    public function buildApiRequest(RequestDataInterface $requestData): Request
    {
        $request = $this->factory->request();
        $request = $this->addAccountHolder($requestData, $request);

        $request->setProcessNumber($requestData->getProcessNumber());
        $request->setTokenId($requestData->getTokenId());
        $request->setAmount($requestData->getAmount());
        $request->setCurrency($requestData->getCurrency());
        $request->setClient($requestData->getClient());

        //add custom fields
        $request = $this->addCustomFields($requestData, $request);

        $violations = $this->validator->validate($request);
        if (0 < count($violations)) {
            $errorMessages = [];
            foreach ($violations as $violation) {
                $errorMessages[] = $violation->getMessage();
            }
            throw new InvalidRequestParametersException(implode(' ', $errorMessages));
        }

        return $request;
    }

    /**
     * @param RequestDataInterface $requestData
     * @param Request $request
     * @return Request
     */
    private function addAccountHolder(RequestDataInterface $requestData, Request $request): Request
    {
        if (null !== $requestData->getFirstName() && null !== $requestData->getLastName()) {
            $accountHolder = $this->factory->accountHolder();
            $accountHolder->setFirstName($requestData->getFirstName());
            $accountHolder->setLastName($requestData->getLastName());
            $request->setAccountHolder($accountHolder);
        }

        return $request;
    }

    /**
     * @param RequestDataInterface $requestData
     * @param Request $request
     * @return Request
     */
    private function addCustomFields(RequestDataInterface $requestData, Request $request): Request
    {
        if (null !== $requestData->getCustomFields()) {
            foreach ($requestData->getCustomFields() ?? [] as $customField) {
                $customFieldAdd = $this->factory->customField();
                $customFieldAdd->setName($customField->getName());
                $customFieldAdd->setValue($customField->getValue());
                $request->addCustomField($customFieldAdd);
            }
        }
        return $request;
    }
}
