<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\PaymentServiceSdk\Tests\Services\Wirecard\PurchaseCreditCard\Request\Version1;

use Doctrine\Common\Collections\ArrayCollection;
use JMS\Serializer\SerializerBuilder;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseCreditCard\Request\Version1\AccountHolder;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseCreditCard\Request\Version1\CustomField;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseCreditCard\Request\Version1\Request;
use PHPUnit\Framework\TestCase;

/**
 * Class RequestTest
 * @package Lifestyle\PaymentServiceSdk\Tests\Services\PurchaseCreditCard\Request\Version1
 */
class RequestTest extends TestCase
{

    public function test_request_model()
    {
        $serializer = SerializerBuilder::create()->build();

        $requestModel = $this->getRequestModel();

        $xml = $serializer->serialize($requestModel, 'xml');
        $this->assertXmlStringEqualsXmlString($this->getXmlString(), $xml);

        $json = $serializer->serialize($requestModel, 'json');
        $this->assertJsonStringEqualsJsonString($this->getJsonString(), $json);

        //also test the getter
        $this->assertEquals('Alan', $requestModel->getAccountHolder()->getFirstName());
        $this->assertEquals('Turing', $requestModel->getAccountHolder()->getLastName());

        $this->assertEquals(4444333322221111, $requestModel->getTokenId());
        $this->assertEquals('EUR', $requestModel->getCurrency());
        $this->assertEquals('1.23', $requestModel->getAmount());
        $this->assertEquals('4711', $requestModel->getProcessNumber());
    }

    /**
     * @return Request
     */
    private function getRequestModel(): Request
    {
        $requestModel = new Request();

        $accountHolder = new AccountHolder();
        $accountHolder->setFirstName('Alan');
        $this->assertEquals('Alan', $accountHolder->getFirstName());

        $accountHolder->setLastName('Turing');
        $this->assertEquals('Turing', $accountHolder->getLastName());
        $requestModel->setAccountHolder($accountHolder);
        $this->assertInstanceOf(AccountHolder::class, $requestModel->getAccountHolder());

        $requestModel->setTokenId(4444333322221111);
        $this->assertEquals('Turing', $accountHolder->getLastName());

        $requestModel->setClient('client');
        $this->assertEquals('client', $requestModel->getClient());

        $requestModel->setCurrency('EUR');
        $this->assertEquals('EUR', $requestModel->getCurrency());

        $requestModel->setAmount(1.23);
        $this->assertEquals(1.23, $requestModel->getAmount());

        $requestModel->setProcessNumber('4711');
        $this->assertEquals('4711', $requestModel->getProcessNumber());

        $customFieldModel = new CustomField();
        $customFieldModel->setName('flight');
        $this->assertEquals('flight', $customFieldModel->getName());
        $customFieldModel->setValue('23233434344');
        $this->assertEquals('23233434344', $customFieldModel->getValue());

        $requestModel->addCustomField($customFieldModel);
        $this->assertInstanceOf(ArrayCollection::class, $requestModel->getCustomFields());

        return $requestModel;
    }

    /**
     * @return string
     */
    private function getXmlString(): string
    {
        return '<?xml version="1.0"?>
                <wirecardPaymentCreditCardRequest>
                  <accountHolder>
                    <firstName>Alan</firstName>
                    <lastName>Turing</lastName>
                  </accountHolder>
                  <processNumber>4711</processNumber>
                  <tokenId>4444333322221111</tokenId>
                  <amount>1.23</amount>
                  <currency>EUR</currency>
                  <client>client</client>
                  <customFields>
                    <entry>
                      <name>flight</name>
                      <value>23233434344</value>
                    </entry>
                  </customFields>
                </wirecardPaymentCreditCardRequest>';
    }

    /**
     * @return string
     */
    private function getJsonString(): string
    {
        return '{
                    "accountHolder": {
                        "firstName": "Alan",
                        "lastName": "Turing"
                    },
                    "processNumber": "4711",
                    "tokenId": 4444333322221111,
                    "amount": 1.23,
                    "currency": "EUR",
                    "client": "client",
                    "customFields": [
                        {
                            "name": "flight",
                            "value": "23233434344"
                        }
                    ]
                }';
    }
}
