<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2020 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseDebit\Processor\Version1;

use GuzzleHttp\Client as HttpClient;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\PaymentServiceSdk\Configuration\ConfigModel;
use Lifestyle\PaymentServiceSdk\Exceptions\InvalidRequestParametersException;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseDebit\Response\Version1\Response;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseDebit\ProcessorInterface;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseDebit\Request\RequestDataInterface;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseDebit\Request\Version1\RequestBuilder;
use Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseDebit\Response\ResponseDataInterface;
use Exception;
use Psr\Log\LoggerInterface;

/**
 * Class Processor
 * @package Lifestyle\PaymentServiceSdk\Services\Wirecard\PurchaseDebit\Processor\Version1
 */
class Processor implements ProcessorInterface
{
    /**
     * @var RequestBuilder
     */
    private $requestBuilder;

    /**
     * @var ConfigModel
     */
    private $config;

    /**
     * @var HttpClient
     */
    private $httpClient;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Processor constructor.
     * @param RequestBuilder $requestBuilder
     * @param ConfigModel $config
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     */
    public function __construct(
        RequestBuilder $requestBuilder,
        ConfigModel $config,
        SerializerInterface $serializer,
        LoggerInterface $logger
    ) {
        $this->requestBuilder = $requestBuilder;
        $this->config = $config;
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->httpClient = new HttpClient();
    }

    /**
     * @param RequestDataInterface $requestData
     * @return ResponseDataInterface
     * @throws InvalidRequestParametersException
     */
    public function process(RequestDataInterface $requestData): ResponseDataInterface
    {
        $request = $this->requestBuilder->buildApiRequest($requestData);
        try {
            $jsonBody = $this->serializer->serialize($request, 'json');
            $responseObject = $this->httpClient->post(
                $this->config->getServiceUrl() . '/api/v1/wirecard/purchase/debit',
                [
                    'headers' => [
                        'X-AUTH-TOKEN' => $this->config->getApiToken(),
                        'Content-Type' => 'application/json;charset=UTF-8'
                    ],
                    'body' => $jsonBody,
                    'timeout' => $this->config->getTimeOut(),
                    'connect_timeout' => $this->config->getConnectionTimeOut()
                ]
            );
            $responseContent = $responseObject->getBody()->getContents();
        } catch (ConnectException $exc) {
            return $this->handleConnectionException($exc);
        } catch (RequestException $exc) {
            return $this->handleRequestException($exc);
        } catch (\Exception $exc) {
            $this->logger->error($exc->getMessage());
            $this->logger->error($exc->getTraceAsString());
            return $this->buildErrorResponse(max((int)$exc->getCode(), 500), 'Internal Server Error');
        }

        try {
            $responseModel = $this->serializer->deserialize($responseContent, Response::class, 'json');
        } catch (Exception $exc) {
            $this->logger->error($exc->getMessage());
            $this->logger->error($exc->getTraceAsString());
            return $this->buildErrorResponse(max((int)$exc->getCode(), 500), 'Internal Server Error');
        }

        return $responseModel;
    }

    /**
     * @param ConnectException $exc
     * @return ResponseDataInterface
     */
    private function handleConnectionException(ConnectException $exc): ResponseDataInterface
    {
        return $this->buildErrorResponse(
            max((int)$exc->getCode(), 400),
            'Wirecard Connection Timeout: ' . $exc->getMessage()
        );
    }

    /**
     * @param RequestException $exc
     * @return ResponseDataInterface
     */
    private function handleRequestException(RequestException $exc): ResponseDataInterface
    {
        if ($exc->getResponse() && $exc->getResponse()->getBody()) {
            try {
                $responseContent = (string)$exc->getResponse()->getBody();
                return $this->serializer->deserialize($responseContent, Response::class, 'json');
            } catch (Exception $exc) {
                return $this->buildErrorResponse(max((int)$exc->getCode(), 500), 'Internal Server Error');
            }

        } else {
            return $this->buildErrorResponse(
                max((int)$exc->getCode(), 400),
                'Wirecard Request Exception: ' . $exc->getMessage()
            );
        }
    }

    /**
     * @param int $code
     * @param string $message
     * @return ResponseDataInterface
     */
    private function buildErrorResponse($code = 0, $message = ''): ResponseDataInterface
    {
        $response = new Response();
        $response->setCode($code);
        $response->setMessage($message);

        return $response;
    }
}
