<?php

/**
 * Class Context
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\ExportBundle\Mapping;

use InvalidArgumentException;
use Lifestyle\Pimcore\ExportBundle\Exception\NoHandlerException;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\ExportBundle\Mapping\Handler\HandlerInterface;

/**
 * Class Context
 *
 * @package Lifestyle\Pimcore\ExportBundle\Mapping
 */
class Context
{
    /**
     * @var HandlerInterface[]|array
     */
    protected $handlers = [];

    /**
     * Registers a new handler in the list
     *
     * @param HandlerInterface $handler
     * @param int              $priority the higher the number the higher the priority
     *
     * @return $this
     * @throws InvalidArgumentException
     */
    public function addHandler(HandlerInterface $handler, $priority = 0)
    {
        if (filter_var($priority, FILTER_VALIDATE_INT) === false) {
            throw new InvalidArgumentException("Argument 'priority' should be an integer, got '$priority'");
        }
        if ($this->handlerAlreadyRegistered($handler)) {
            return $this;
        }
        $this->handlers[$priority][] = $handler;

        return $this;
    }

    /**
     * Sorts the priority list but leaves the original insertion order for clashes untouched.
     * This ensures that we adhere to FIFO within each priority level.
     *
     * Priority is sorted from HIGH to LOW
     */
    protected function sortHandlers()
    {
        krsort($this->handlers);
    }

    /**
     * Gets a single (first) handler compatible with this input
     *
     * @param mixed $input
     *
     * @return HandlerInterface
     * @throws NoHandlerException
     */
    public function getHandlerFor($input)
    {
        $allHandlers = $this->getAllHandlersFor($input);

        return array_shift($allHandlers);
    }

    /**
     * Retrieves all the handlers that can handle this input
     *
     * @param mixed $input
     *
     * @return HandlerInterface[]
     * @throws NoHandlerException
     */
    public function getAllHandlersFor($input)
    {
        $this->sortHandlers();
        $filteredIterator = new \CallbackFilterIterator(
            $this->getHandlerIterator(),
            function (HandlerInterface $handler) use ($input) {
                return $handler->handles($input);
            }
        );
        $compatibleHandlers = iterator_to_array($filteredIterator, false);
        if (empty($compatibleHandlers)) {
            throw new NoHandlerException();
        }

        return $compatibleHandlers;
    }

    /**
     * Handlers input by delegating to the proper handler
     *
     * @param DataCollectorInterface $collector
     * @param mixed $object
     * @param        $property
     * @param string $language
     */
    public function handle(DataCollectorInterface $collector, $object, $property, $language)
    {
        $this->getHandlerFor($property)->handle($collector, $object, $property, $language);
    }

    /**
     * Checks if handler is already registered
     *
     * @param HandlerInterface $handler
     *
     * @return bool
     */
    public function handlerAlreadyRegistered(HandlerInterface $handler)
    {
        foreach ($this->getHandlerIterator() as $registeredHandler) {
            if ($registeredHandler === $handler) {
                return true;
            }
        }

        return false;
    }

    /**
     * Sets up and returns a recursive handler iterator.
     *
     * @return \RecursiveIteratorIterator
     */
    protected function getHandlerIterator()
    {
        return new \RecursiveIteratorIterator(
            new \RecursiveArrayIterator($this->handlers, \RecursiveArrayIterator::CHILD_ARRAYS_ONLY),
            \RecursiveIteratorIterator::LEAVES_ONLY
        );
    }
}
