<?php

/**
 * Class DataCollectorTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector;

use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollector;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollectorItem;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollectorItemInterface;
use PHPUnit\Framework\TestCase;

/**
 * Class DataCollectorTest
 * @package Tests\Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector
 */
class DataCollectorTest extends TestCase
{
    public function testAddStandardItem()
    {
        $itemKey = 'standard_item';
        $itemValue = 'standard_item_value';

        $collector = new DataCollector();
        $collector->addItem($itemKey, $itemValue);

        $this->assertTrue($collector->hasItem($itemKey));
        $this->assertInstanceOf(DataCollectorItem::class, $collector->getItem($itemKey));
        $this->assertInstanceOf(DataCollectorItemInterface::class, $collector->getItem($itemKey));
        $this->assertSame($itemValue, $collector->getItemValue($itemKey));
        $this->assertFalse($collector->getItem($itemKey)->hasLocale());
        $this->assertCount(1, $collector->getItems());
    }

    public function testAddTranslatedItem()
    {
        $itemKey = 'translated_item';
        $itemValueDE = 'translated_item_DE';
        $itemValueEN = 'translated_item_EN';

        $localeDE = 'de_DE';
        $localeEN = 'en_EN';

        $collector = new DataCollector();
        $collector->addItem($itemKey, $itemValueDE, $localeDE);
        $collector->addItem($itemKey, $itemValueEN, $localeEN);

        // Assertions for first $localeDE
        $this->assertTrue($collector->hasItem($itemKey, $localeDE));
        $this->assertInstanceOf(DataCollectorItem::class, $collector->getItem($itemKey, $localeDE));
        $this->assertInstanceOf(DataCollectorItemInterface::class, $collector->getItem($itemKey, $localeDE));
        $this->assertSame($itemValueDE, $collector->getItemValue($itemKey, $localeDE));
        $this->assertTrue($collector->getItem($itemKey, $localeDE)->hasLocale());
        $this->assertCount(2, $collector->getItems());
    }

    public function testResetTranslatedItem()
    {
        $itemKey = 'standard_item';
        $itemValue = 'standard_item_value';

        $collector = new DataCollector();
        $collector->addItem($itemKey, $itemValue);

        $this->assertTrue($collector->hasItem($itemKey));
        $this->assertInstanceOf(DataCollectorItem::class, $collector->getItem($itemKey));
        $this->assertInstanceOf(DataCollectorItemInterface::class, $collector->getItem($itemKey));
        $this->assertSame($itemValue, $collector->getItemValue($itemKey));
        $this->assertFalse($collector->getItem($itemKey)->hasLocale());
        $this->assertCount(1, $collector->getItems());

        $collectorItem = $collector->getItem($itemKey);
        $collectorItem->setValue('new_value');
        $collectorItem->setLocale('new_locale');

        $this->assertSame('new_value', $collector->getItemValue($itemKey));
        $this->assertSame('new_locale', $collector->getItem($itemKey)->getLocale());
    }

    public function testGetItemWillReturnNull()
    {
        $collector = new DataCollector();
        $collector->addItem('some_item_key', 'some_item_value');

        $this->assertNull($collector->getItem('not_existing_item_key'));
    }

    public function testGetItemValueWillReturnNull()
    {
        $collector = new DataCollector();
        $collector->addItem('some_item_key', 'some_item_value');

        $this->assertNull($collector->getItemValue('not_existing_item_key'));
    }

    public function testGetItemsByLocale()
    {
        $localeDE = 'de_DE';
        $localeEN = 'en_EN';

        $collector = new DataCollector();
        $collector->addItem('de_DE_key_1', 'de_DE_value_1', $localeDE);
        $collector->addItem('de_DE_key_2', 'de_DE_value_2', $localeDE);
        $collector->addItem('de_DE_key_3', 'de_DE_value_3', $localeDE);

        $collector->addItem('en_EN_key_1', 'en_EN_value_1', $localeEN);
        $collector->addItem('en_EN_key_2', 'en_EN_value_2', $localeEN);
        $collector->addItem('en_EN_key_3', 'en_EN_value_3', $localeEN);

        $this->assertCount(3, $collector->getItems($localeDE));
        $this->assertCount(3, $collector->getItems($localeEN));
        $this->assertSame([], $collector->getItems('unknown_LOCALE'));
    }

    public function testGetCollectedLocales()
    {
        $collector = new DataCollector();
        $collector->addItem('de_DE_key', 'de_DE_value', 'de_DE');
        $collector->addItem('en_DE_key', 'en_DE_value', 'en_DE');
        $collector->addItem('en_US_key', 'en_US_value', 'en_US');
        $collector->addItem('en_EN_key', 'en_EN_value', 'en_EN');

        $collectedLocales = $collector->getCollectedLocales();

        $this->assertCount(4, $collectedLocales);
        $this->assertSame('de_DE', $collectedLocales[0]);
        $this->assertSame('en_DE', $collectedLocales[1]);
        $this->assertSame('en_US', $collectedLocales[2]);
        $this->assertSame('en_EN', $collectedLocales[3]);
    }

    public function testAddExistingItemWillDoNothing()
    {
        // Test without locale
        $collector = new DataCollector();
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value');
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value');

        $this->assertCount(1, $collector->getItems());

        // Test with locale
        $collector = new DataCollector();
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value', 'en_US');
        $collector->addItem('unit_test_unique_key', 'unit_test_unique_value', 'en_US');

        $this->assertCount(1, $collector->getItems());
    }
}
