# Pimcore5 Export
This is the base plugin to map general pimcore5 fields (input, checkbox, number, etc.) and to trigger registered export handlers in order to distribute data to other services.

To register export handlers, just install one of the available export handler plugins for pimcore5:
 - **RabbitMQ** - https://stash.life-style.de/projects/PIM/repos/pimcore5-rabbitmq
 - **Sylius** - https://stash.life-style.de/projects/PIM/repos/pimcore5-sylius
 - **Swoox** - https://stash.life-style.de/projects/PIM/repos/pimcore5-swoox

**The following field types will be mapped:**
- Checkbox
- ClassificationStore
- Href (for Asset Objects)
- Image
- ImageGallery
- Input
- Localized
- Number
- ObjectList (Single Object ID or list of Object IDs)
- Select

## Installation


### Dependencies

This package requires another package `lifestyle/data-collector-model` which also uses a repository,
which is not in the list of official repositories. Due to a
[composer limitation](https://getcomposer.org/doc/faqs/why-can%27t-composer-load-repositories-recursively.md)
the depended repositories has to be added to your application's composer.json.


### Prerequisites

Make sure you have added the repository in your composer.json file:

```json
    {
        "minimum-stability": "dev",
        "prefer-stable": true,
        "repositories": [
            {
                  "type": "vcs",
                  "url": "ssh://git@stash.life-style.de:7999/pim/pimcore5-export.git"
            },
            {
              "type": "vcs",
              "url": "ssh://git@stash.life-style.de:7999/sb/data-collector-model.git"
            }
        ]
    }
```


### Installation using composer
```
    composer require lifestyle/pimcore-export dev-develop
```

If the plugin has been installed successfully, you can enable it in the pimcore extension manager.


## Cookbook

### Add a new general Field Mapper Service
- Create a new class which implements `FieldMapperInterface`.
- Define your conditions in the `handles` method weather the mapper is applicable or not.
- Implement your mapping logic in the `handle` method. There you can append data to the DataCollector which `maintains` all mapped data at the end.
- Tag your service with `lifestyle_pimcore_export.field_mapper`


Here is an example how to do that for a basic pimcore input field:
 

```php

namespace Lifestyle\Pimcore\ExportBundle\Mapping\Field;
 
use Pimcore\Model\DataObject\ClassDefinition\Data;
use Pimcore\Model\DataObject\ClassDefinition\Data\Input;
use Pimcore\Model\DataObject\Concrete;
use Lifestyle\DataCollector\DataCollectorInterface;
 
class InputFieldMapper implements FieldMapperInterface
{
    public function handles(Data $property)
    {
        return $property instanceof Input;
    }
 
    public function handle(DataCollectorInterface $collector, Concrete $object, Data $property, $language)
    {
        $collector->addItem($property->getName(), $object->getValueForFieldName($property->getName()));
    }
}

```

### Add a custom Field Mapper Service
The service must be tagged with `lifestyle_pimcore_export.field_mapper`. It is recommended to add a priority to the tag of the service (`priority: 1`).
If this is done, the service will be loaded automatically independent of it's location.
Therefor, you can place your service with customer specific logic everywhere you want, for example, own composer packages or symfony bundles.  


### Add a custom Object Mapper
By default the `DefaultMapper` will map all fields for your Pimcore Object.
In some special cases you might want to do additional things while mapping the data.
For example, you have product variants which should inherit some data from base products.
To solve this, add a new object mapper:
- Create a new class which implements `MapperInterface`.
- Define your conditions in the `applicable` method weather the mapper is applicable or not.
- Implement your custom mapping workflow in the methods suggested by the interface.
- Tag your service with `lifestyle_pimcore_export.object_mapper`. Be sure to set the priority at least to `1`! Otherwise it could be that your service won't be executed because the default mapper is prioritized higher. 

Example Object Mapper:

```php
use Lifestyle\Pimcore\ExportBundle\Mapping\Objects\MappingProcessor;
use Lifestyle\DataCollector\DataCollectorInterface;

 
class CustomMapper implements MapperInterface
{
    /**
     * @var MappingProcessor
     */
    private $mappingProcessor;
 
    public function __construct(MappingProcessor $mappingProcessor)
    {
        $this->mappingProcessor = $mappingProcessor;
    }
 
    public function applicable($object)
    {
        // Apply mapper for ProductVariant objects.
        return $object instanceof ProductVariant;
    }
 
    /**
     * @return DataCollectorInterface
     */
    public function getUpdateDataCollector($object)
    {
        // Here you might want to add the logic for data inheritance, for example.
        return $this->mappingProcessor->mapEverythingFromObject($object);
    }
 
    /**
     * @return DataCollectorInterface
     */
    public function getDeleteDataCollector($object)
    {
        return $this->mappingProcessor->mapBaseFromObject($object);
    }
}

```

Of course, for a tagged service it doesn't matter, where the service is placed.
That gives you the opportunity to overwrite default behavior and to provide custom bundles/packages easily.  


### Add a new export handler
- Create a new class which implements `Lifestyle\Pimcore\ExportBundle\Export\HandlerInterface`
- Add the methods suggested by the interface
- Method `isResponsible` defines, weather your export handler is responsible or not.
- Tag your service with `lifestyle_pimcore_export.export_handler`

Each registered export handler will be executed.

Here is an example:
```php
namespace Lifestyle\Pimcore\SwooxBundle\Export;
 
use GuzzleHttp\ClientInterface;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\ExportBundle\Export\HandlerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Log\LoggerInterface;
 
/**
 * Class Handler
 * @package Lifestyle\Pimcore\SwooxBundle\Export
 */
class Handler implements HandlerInterface
{
    const HANDLER_IDS = ['Product', 'Variant'];
 
    /**
     * @var ClientInterface
     */
    private $client;
 
    /**
     * @var SerializerInterface
     */
    private $serializer;
 
    /**
     * @var LoggerInterface
     */
    private $logger;
 
    /**
     * @var Mapper
     */
    private $mapper;
 
    /**
     * @var Config
     */
    private $config;
 
    // ...
     
    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return in_array($collector->getItemValue('className'), self::HANDLER_IDS);
    }
 
    /**
     * @param DataCollectorInterface $collector
     */
    public function publish(DataCollectorInterface $collector)
    {
        $requestModel = $this->mapper->mapUpdateRequestModel($collector);
        $this->send('PUT', $this->config->getIndexUrl(), $requestModel);
    }
 
    /**
     * @param DataCollectorInterface $collector
     */
    public function unpublish(DataCollectorInterface $collector)
    {
        $requestModel = $this->mapper->mapDeleteRequestModel($collector);
        $this->send('DELETE', $this->config->getDeleteUrl(), $requestModel);
    }
     
    //...
}
```


## Additional documentation

https://intranet.life-style.de/display/MYS/Pimcore+Export


Have fun!
