<?php

/**
 * Class MappingProcessor
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\ExportBundle\Mapping\Objects;

use Lifestyle\Pimcore\ExportBundle\Mapping\Field\FieldMapperProcessor;
use Pimcore\Config;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\Concrete;
use Lifestyle\Pimcore\ExportBundle\Exception\NoHandlerException;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\DataCollector\DataCollectorInterface;
use Psr\Log\LoggerInterface;

/**
 * Class MappingProcessor
 *
 * @package Lifestyle\Pimcore\ExportBundle\Mapping\Objects
 */
class MappingProcessor
{
    /**
     * @var FieldMapperProcessor
     */
    private $fieldMapperProcessor;

    /**
     * @var array
     */
    private $locales;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param FieldMapperProcessor $fieldMapperProcessor
     * @param LoggerInterface $logger
     * @throws \Exception
     */
    public function __construct(FieldMapperProcessor $fieldMapperProcessor, LoggerInterface $logger)
    {
        $this->fieldMapperProcessor = $fieldMapperProcessor;
        $this->logger = $logger;

        $config = Config::getSystemConfig();
        $languages = explode(',', $config->general->validLanguages);

        $this->locales = $languages ? $languages : [];
    }

    /**
     * @param AbstractObject $object
     * @param string $eventName
     * @return DataCollectorInterface
     */
    public function mapEverythingFromObject($object, $eventName): DataCollectorInterface
    {
        $collector = $this->mapBaseFromObject($object, $eventName);

        foreach ($this->locales as $lang) {

            /**
             * @var \Pimcore\Model\DataObject\ClassDefinition\Data[] $property
             */
            foreach ($object->getClass()->getFieldDefinitions() as $property) {
                try {
                    $this->fieldMapperProcessor->handle($collector, $object, $property, $lang);
                } catch (NoHandlerException $exception) {
                    $this->logger->info($exception->getMessage());
                }
            }
        }

        return $collector;
    }

    /**
     * @param \Pimcore\Model\DataObject\Concrete $object
     * @param string $eventName
     * @return DataCollector
     */
    public function mapBaseFromObject($object, $eventName): DataCollectorInterface
    {
        $collector = new DataCollector();
        $collector->addItem('id', $object->getId());
        $collector->addItem('className', $object->getClassName());
        $collector->addItem('event', strtolower($object->getClassName() . '.' . $eventName));
        $collector->addItem('locales', $this->locales);

        if (null !== ($parent = $this->getNextParentForInheritance($object))) {
            $collector->addItem('parentId', $parent->getId());
            $collector->addItem('parentClassName', $parent->getClassName());
        }

        $collector->addItem('type', $object->getType());

        return $collector;
    }

    /**
     * @param Concrete $object
     * @param string|null $parentClass
     * @return AbstractObject|null
     */
    public function getNextParentForInheritance(Concrete $object, $parentClass = null)
    {
        if ($object instanceof AbstractObject) {
            $parent = $object->getParent();
            while ($parent && $parent->getType() == 'folder') {
                $parent = $parent->getParent();
            }

            if (
                $parent &&
                ($parent->getType() == 'object' || $parent->getType() == 'variant') &&
                (
                    !$parentClass || ($parentClass && $parent->getClassName() == $parentClass)
                )
            ) {
                return $parent;
            }
        }

        return null;
    }
}
