<?php

/**
 * Class FieldMapperPassTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\ExportBundle\Compiler;

use Lifestyle\Pimcore\ExportBundle\Compiler\FieldMapperPass;
use Lifestyle\Pimcore\ExportBundle\Exception\InvalidFieldMapperException;
use Lifestyle\Pimcore\ExportBundle\Mapping\Field\FieldMapperContext;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Class FieldMapperPassTest
 * @package Tests\Lifestyle\Pimcore\ExportBundle\Compiler
 */
class FieldMapperPassTest extends TestCase
{
    public function testBaseDefinitionNotFound()
    {
        $container = new ContainerBuilder();

        $pass = new FieldMapperPass();
        $this->assertNull($pass->process($container));
    }

    public function testNoTaggedServices()
    {
        $container = new ContainerBuilder();

        $container
            ->register('lifestyle_pimcore_export.mapping.field.field_mapper_context')
            ->setClass(FieldMapperContext::class);

        $pass = new FieldMapperPass();
        $pass->process($container);

        $context = $container->get('lifestyle_pimcore_export.mapping.field.field_mapper_context');
        $this->assertAttributeCount(0, 'handlers', $context);
    }

    public function testTaggedServicesWillBeAddedByPriority()
    {
        $container = new ContainerBuilder();

        $container
            ->register('lifestyle_pimcore_export.mapping.field.field_mapper_context')
            ->setClass(FieldMapperContext::class);

        $container
            ->register('mapper.last')
            ->setClass(FieldMapperMock::class)
            ->setProperty('priority', 'last')
            ->addTag('lifestyle_pimcore_export.field_mapper', ['priority' => 2])
            ->setPublic(false);

        $container
            ->register('mapper.first')
            ->setClass(FieldMapperMock::class)
            ->setProperty('priority', 'first')
            ->addTag('lifestyle_pimcore_export.field_mapper')
            ->setPublic(false);

        $container
            ->register('mapper.second')
            ->setClass(FieldMapperMock::class)
            ->setProperty('priority', 'second')
            ->addTag('lifestyle_pimcore_export.field_mapper', ['priority' => 1])
            ->setPublic(false);

        $container
            ->register('mapper.third')
            ->setClass(FieldMapperMock::class)
            ->setProperty('priority', 'third')
            ->addTag('lifestyle_pimcore_export.field_mapper', ['priority' => 1])
            ->setPublic(false);

        $pass = new FieldMapperPass();
        $pass->process($container);

        $context = $container->get('lifestyle_pimcore_export.mapping.field.field_mapper_context');

        $reflector = new \ReflectionClass($context);
        $property = $reflector->getProperty('handlers');
        $property->setAccessible(true);

        $fieldMapperHandler = $property->getValue($context);
        $this->assertCount(3, $fieldMapperHandler);
        $this->assertCount(1, $fieldMapperHandler[2]);

        $this->assertSame('first', $fieldMapperHandler[0][0]->getPriority());
        $this->assertSame('second', $fieldMapperHandler[1][0]->getPriority());
        $this->assertSame('third', $fieldMapperHandler[1][1]->getPriority());
        $this->assertSame('last', $fieldMapperHandler[2][0]->getPriority());

        foreach ($property->getValue($context)[0] as $handler) {
            $this->assertInstanceOf(FieldMapperMock::class, $handler);
        }
    }

    public function testProcessWillThrowMissingInterfaceException()
    {
        $this->expectException(InvalidFieldMapperException::class);

        $container = new ContainerBuilder();

        $container
            ->register('lifestyle_pimcore_export.mapping.field.field_mapper_context')
            ->setClass(FieldMapperContext::class);

        $container
            ->register('handler.foo')
            ->addTag('lifestyle_pimcore_export.field_mapper')
            ->setPublic(false);

        $pass = new FieldMapperPass();
        $pass->process($container);
    }
}
