<?php

/**
 * Class MappingProcessor
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\ExportBundle\Mapping\Objects;

use Lifestyle\Pimcore\ExportBundle\Configuration\LocaleConfiguration;
use Lifestyle\Pimcore\ExportBundle\Mapping\Field\FieldMapperProcessor;
use Pimcore\Config;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\Concrete;
use Lifestyle\Pimcore\ExportBundle\Exception\NoHandlerException;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\DataCollector\DataCollectorInterface;
use Pimcore\Model\DataObject\Folder;
use Psr\Log\LoggerInterface;

/**
 * Class MappingProcessor
 *
 * @package Lifestyle\Pimcore\ExportBundle\Mapping\Objects
 */
class MappingProcessor
{
    /**
     * @var FieldMapperProcessor
     */
    private $fieldMapperProcessor;

    /**
     * @var LocaleConfiguration
     */
    private $localeConfiguration;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param FieldMapperProcessor $fieldMapperProcessor
     * @param LocaleConfiguration $localeConfiguration
     * @param LoggerInterface $logger
     */
    public function __construct(
        FieldMapperProcessor $fieldMapperProcessor,
        LocaleConfiguration $localeConfiguration,
        LoggerInterface $logger
    ) {
        $this->fieldMapperProcessor = $fieldMapperProcessor;
        $this->localeConfiguration = $localeConfiguration;
        $this->logger = $logger;
    }

    /**
     * @param AbstractObject $object
     * @param string $eventName
     * @return DataCollectorInterface
     */
    public function mapEverythingFromObject($object, $eventName): DataCollectorInterface
    {
        $collector = $this->mapBaseFromObject($object, $eventName);

        foreach ($this->localeConfiguration->getLanguages() as $lang) {

            /**
             * @var \Pimcore\Model\DataObject\ClassDefinition\Data[] $property
             */
            foreach ($object->getClass()->getFieldDefinitions() as $property) {
                try {
                    $this->fieldMapperProcessor->handle($collector, $object, $property, $lang);
                } catch (NoHandlerException $exception) {
                    $this->logger->info($exception->getMessage());
                }
            }
        }

        return $collector;
    }

    /**
     * @param \Pimcore\Model\DataObject\Concrete $object
     * @param string $eventName
     * @return DataCollector
     */
    public function mapBaseFromObject($object, $eventName): DataCollectorInterface
    {
        $collector = new DataCollector();
        $collector->addItem('id', $object->getId());
        $collector->addItem('className', $object->getClassName());
        $collector->addItem('event', strtolower($object->getClassName() . '.' . $eventName));

        $localeConfig = $this->localeConfiguration;
        $collector->addItem(
            'locales',
            $localeConfig->usePrimaryLanguages() ? $localeConfig->getPrimaryLanguages() : $localeConfig->getLanguages()
        );

        if (null !== ($parent = $this->getNextParentForInheritance($object))) {
            $collector->addItem('parentId', $parent->getId());
            $collector->addItem('parentClassName', $parent->getClassName());
        }

        $collector->addItem('type', $object->getType());

        $collector->addItem('folder', $object->getPath());
        $folder = $this->getObjectFolder($object);
        if (null !== $folder) {
            $collector->addItem('folderId', $folder->getId());
        }

        return $collector;
    }

    /**
     * @param Concrete $object
     * @param string|null $parentClass
     * @return AbstractObject|null
     */
    public function getNextParentForInheritance(Concrete $object, $parentClass = null)
    {
        if ($object instanceof AbstractObject) {
            $parent = $object->getParent();
            while ($parent && $parent->getType() === AbstractObject::OBJECT_TYPE_FOLDER) {
                $parent = $parent->getParent();
            }

            if (
                $parent &&
                (
                    $parent->getType() === AbstractObject::OBJECT_TYPE_OBJECT ||
                    $parent->getType() === AbstractObject::OBJECT_TYPE_VARIANT
                ) &&
                (
                    !$parentClass ||
                    $parentClass && $parent->getClassName() == $parentClass
                )
            ) {
                return $parent;
            }
        }

        return null;
    }

    /**
     * @param AbstractObject $object
     * @return Folder|null
     */
    private function getObjectFolder(AbstractObject $object)
    {
        while (null !== ($object = $object->getParent())) {
            if (AbstractObject::OBJECT_TYPE_FOLDER === $object->getType()) {
                return $object;
            }
        }
        return null;
    }
}
