<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\ExportBundle\Mapping\Field;

use Lifestyle\Pimcore\ExportBundle\Configuration\LocaleConfiguration;
use Pimcore\Model\DataObject\ClassDefinition\Data;
use Pimcore\Model\DataObject\ClassDefinition\Data\Localizedfields;
use Pimcore\Model\DataObject\Concrete;
use Lifestyle\DataCollector\DataCollectorInterface;

/**
 * Class LocalizedFieldMapper
 * @package Lifestyle\Pimcore\ExportBundle\Mapping\Field
 */
class LocalizedFieldMapper implements FieldMapperInterface
{
    /**
     * @var LocaleConfiguration
     */
    private $localeConfiguration;

    /**
     * LocalizedFieldMapper constructor.
     * @param LocaleConfiguration $localeConfiguration
     */
    public function __construct(LocaleConfiguration $localeConfiguration)
    {
        $this->localeConfiguration = $localeConfiguration;
    }

    /**
     * @param Data $property
     * @return bool
     */
    public function handles(Data $property)
    {
        return $property instanceof Localizedfields;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param Concrete $object
     * @param Data $property
     * @param string $language
     */
    public function handle(DataCollectorInterface $collector, Concrete $object, Data $property, $language)
    {
        /** @var \Pimcore\Model\DataObject\ClassDefinition\Data\Localizedfields $property */

        $locale = $this->localeConfiguration->usePrimaryLanguages() ? \Locale::getPrimaryLanguage($language) : $language;
        $parent = $object->getNextParentForInheritance();

        // Retrieve first localized fields block
        foreach ($property->getChildren() as $field) {
            $propertyName = $field->getName();
            $value = $object->getLocalizedfields()->getLocalizedValue($propertyName, $language);
            if ((false === $value || null === $value) && null !== $parent) {
                $value = $parent->getLocalizedfields()->getLocalizedValue($propertyName, $language);
            }
            if (false !== $value && null !== $value) {
                $collector->addItem($propertyName, $value, $locale);
            }
        }

        $processed = [];

        // Pimcore provides further localized fields blocks as referencedFields.
        foreach ($property->getReferencedFields() as $child) {
            foreach ($child->getChildren() as $field) {
                $propertyName = $field->getName();
                if (in_array($propertyName, $processed)) {
                    break;
                }

                $value = $object->getLocalizedfields()->getLocalizedValue($propertyName, $language);
                if ((false === $value || null === $value) && null !== $parent) {
                    $value = $parent->getLocalizedfields()->getLocalizedValue($propertyName, $language);
                }
                if (false !== $value && null !== $value) {
                    $collector->addItem($propertyName, $value, $locale);
                }

                $processed[] = $field->getName();
            }
        }
    }
}
