<?php

/**
 * Class Object
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\ExportBundle\Listener;

use Pimcore\Event\Model\DataObjectEvent;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\Element\ValidationException;
use Psr\Log\LoggerInterface;
use Lifestyle\Pimcore\ExportBundle\Mapping\Object\Handler;
use Symfony\Component\VarDumper\Cloner\VarCloner;
use Symfony\Component\VarDumper\Dumper\CliDumper;
use Symfony\Component\VarDumper\VarDumper;

/**
 * Class Object
 *
 * @package Lifestyle\Pimcore\ExportBundle\Action
 */
class ObjectLS
{
    /**
     * @var \Pimcore\Log\ApplicationLogger|LoggerInterface
     */
    private $logger;

    /**
     * @var Handler
     */
    private $objectMapper;

    /**
     * @var bool
     */
    private $enabled;

    /**
     * Object constructor.
     * @param LoggerInterface $logger
     * @param Handler $objectMapper
     * @throws ValidationException
     */
    public function __construct(LoggerInterface $logger, Handler $objectMapper, bool $enabled)
    {
        $this->logger = $logger;
        $this->objectMapper = $objectMapper;
        $this->enabled = $enabled;

        if (class_exists('VarDumper')) {
            VarDumper::setHandler(function ($var) use ($logger) {

                $cloner = new VarCloner();
                $dumper = new CliDumper();
                $output = fopen('php://memory', 'r+b');

                $dumper->dump($cloner->cloneVar($var), $output);
                $logger->debug(stream_get_contents($output, -1, 0));
            });
        }
    }

    /**
     * @param DataObjectEvent $event
     * @throws ValidationException
     */
    public function onPostUpdate(DataObjectEvent $event)
    {
        if (!$this->enabled || !$event->getObject() instanceof Concrete) {
            return;
        }

        $this->objectMapper->validateProductOptions($event);

        try {
            $this->objectMapper->update($event);
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage());
            throw new ValidationException($exception->getMessage());
        }
    }

    /**
     * @param DataObjectEvent $event
     * @throws ValidationException
     */
    public function onPreDelete(DataObjectEvent $event)
    {
        if (!$this->enabled || !$event->getObject() instanceof Concrete) {
            return;
        }

        /** @var Concrete $object */
        $object = $event->getObject();
        $originalPublishFlag = $object->getPublished();

        try {
            $this->objectMapper->delete($event);
        } catch (\Exception $exception) {
            // Republish objects to prevent invalid object states.
            if (true === $originalPublishFlag) {
                $object->setPublished($originalPublishFlag);
            }

            $this->logger->error($exception->getMessage());
            throw new ValidationException($exception->getMessage());
        }
    }
}
