<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\ExportBundle\Mapping\Object;

use Lifestyle\Pimcore\ExportBundle\Export\Processor;
use Pimcore\Event\Model\DataObjectEvent;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\DataObject\Product;
use Pimcore\Model\DataObject\Variant;
use Pimcore\Model\Element\ValidationException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 *
 * @package Lifestyle\Pimcore\ExportBundle\Mapping\Object
 */
class Handler
{
    /**
     * @var MapperInterface[]|array
     */
    private $availableObjectMapper = [];

    /**
     * @var Processor
     */
    private $exportProcessor;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Handler constructor.
     *
     * @param Processor $exportProcessor
     * @param LoggerInterface $logger
     */
    public function __construct(Processor $exportProcessor, LoggerInterface $logger)
    {
        $this->exportProcessor = $exportProcessor;
        $this->logger = $logger;
    }

    /**
     * @param MapperInterface $objectMapper
     */
    public function addObjectMapper(MapperInterface $objectMapper)
    {
        $this->availableObjectMapper[] = $objectMapper;
    }

    /**
     * update
     *
     * @param DataObjectEvent $event
     */
    public function update(DataObjectEvent $event)
    {
        if (!$event->hasArgument('saveVersionOnly') || $event->getArgument('saveVersionOnly') !== true) {
            /**
             * @var Concrete $object
             */
            $object = $event->getObject();

            /**
             * check for save version only!
             */
            foreach ($this->availableObjectMapper as $mapper) {
                if ($mapper->applicable($object)) {
                    $this->logger->debug(
                        sprintf(
                            "Active Update Object Mapper: %s",
                            (new \ReflectionClass($mapper))->getShortName()
                        )
                    );

                    $collector = $mapper->getUpdateDataCollector($object);

                    if (!$object->isPublished()) {
                        $collector = $mapper->getDeleteDataCollector($object);
                        $this->exportProcessor->delete($collector);
                    } else {
                        $this->exportProcessor->createOrUpdate($collector);
                    }

                    $this->logger->debug(print_r($collector, true));
                    return;
                }
            }
        }
    }

    /**
     * @param DataObjectEvent $event
     * @throws ValidationException
     */
    public function validateProductOptions(DataObjectEvent $event)
    {
        if (!$event->hasArgument('saveVersionOnly') || $event->getArgument('saveVersionOnly') !== true) {
            /**
             * @var Concrete $object
             */
            $object = $event->getObject();
            if ($object->isPublished()) {
                // check if product options from product & variants are the same
                if ('Product' === $object->getClassName()) {
                    $this->validateVariants($object);
                } elseif ('Variant' === $object->getClassName()) {
                    $this->validateProduct($object);
                }
            }
        }
    }

    /**
     * @param Product $product
     * @throws ValidationException
     */
    private function validateVariants(Product $product)
    {
        // Skip validation for projects without class variant
        if (!class_exists('\Pimcore\Model\DataObject\Variant')) {
            return;
        }

        $productOptions = array_keys($product->getOptions()->getItems());

        /** @var Variant $variant */
        foreach ($product->getChildren() as $variant) {

            // Pimcore allows any type beneath an object - skip everything which is not a variant
            if (!$variant instanceof Variant) {
                continue;
            }

            $variantOptions = array_keys($variant->getOptionValues()->getItems());

            // check options
            if (
                $variant->isPublished() &&
                (
                    count($productOptions) != count($variantOptions) ||
                    count(array_diff($productOptions, $variantOptions))
                ) > 0
            ) {
                throw new ValidationException(
                    sprintf("Product options may not differ from variant options, if a variant is published with these options (Variant ID: %s) Solution: Unpublish variant, publish current product, then re-publish the variant with the same options.",
                        $variant->getId()
                    )
                );
            }
        }
    }

    /**
     * @param Variant $variant
     * @throws ValidationException
     */
    private function validateProduct(Variant $variant)
    {
        $variantOptions = array_keys($variant->getOptionValues()->getItems());
        $productOptions = array_keys($variant->getParent()->getOptions()->getItems());

        if (
            count($productOptions) != count($variantOptions) ||
            count(array_diff($productOptions, $variantOptions)) > 0
        ) {
            throw new ValidationException("Variant options may not differ from product options");
        }
    }

    /**
     * @param DataObjectEvent $event
     */
    public function delete(DataObjectEvent $event)
    {
        if (!$event->hasArgument('saveVersionOnly') || $event->getArgument('saveVersionOnly') !== true) {
            /**
             * @var Concrete $object
             */
            $object = $event->getObject();

            /**
             * check for save version only!
             */
            foreach ($this->availableObjectMapper as $mapper) {
                if ($mapper->applicable($object)) {
                    $this->logger->debug(
                        sprintf(
                            "Active Delete Object Mapper: %s",
                            (new \ReflectionClass($mapper))->getShortName()
                        )
                    );

                    $collector = $mapper->getDeleteDataCollector($object);
                    $this->exportProcessor->delete($collector);

                    $this->logger->debug(print_r($collector, true));
                    return;
                }
            }
        }
    }
}
