<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Pimcore\LayoutChangerBundle\EventListener;

use Lifestyle\Pimcore\LayoutChangerBundle\Configuration\CustomViewList;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\GenericEvent;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition\CustomLayout;
use Pimcore\Model\DataObject\Service;
use Pimcore\Tool\Admin;

/**
 * Class ObjectVariantsLayoutChangeListener
 * @package Lifestyle\Pimcore\ObjectDynamicCustomLayoutChangerBundle\EventListener
 */
class ObjectVariantsLayoutChangeListener
{
    /**
     * @var CustomViewList
     */
    private $customViewList;

    /**
     * ObjectVariantsLayoutChangeListener constructor.
     *
     * @param CustomViewList $customViewList
     */
    public function __construct(CustomViewList $customViewList)
    {
        $this->customViewList = $customViewList;
    }

    /**
     * @desc set the custom layout for variant products / reduced fields
     * @param GenericEvent $event
     * @throws \Exception
     */
    public function selectCustomLayout(GenericEvent $event)
    {
        /** @var AbstractObject $object */
        $object = $event->getArgument("object");
        if(false === Admin::getCurrentUser()->isAdmin()
            && in_array($object->getType(), [AbstractObject::OBJECT_TYPE_VARIANT, AbstractObject::OBJECT_TYPE_OBJECT])
            && $this->customViewList->hasCustomViewConfig($object->getClassName())
        ) {
            $customViewConfig = $this->customViewList->getCustomViewConfig($object->getClassName());
            $customLayoutToSelect = $customViewConfig->getObjectViewId();
            $customLayoutToRemove = $customViewConfig->getVariantViewId();
            if (AbstractObject::OBJECT_TYPE_VARIANT === $object->getType()) {
                $customLayoutToSelect = $customViewConfig->getVariantViewId();
                $customLayoutToRemove = $customViewConfig->getObjectViewId();
            }

            $event->setArgument(
                "data",
                $this->doModifyCustomLayouts($event->getArgument("data"), $object, $customLayoutToSelect, [0, -1, $customLayoutToRemove])
            );
        }
    }

    /**
     * @desc Modify available custom layouts according to needs
     *
     * @param       $data
     * @param       $object
     * @param null  $customLayoutToSelect
     * @param array $layoutsToRemove
     *
     * @return mixed
     * @throws \Exception
     */
    private function doModifyCustomLayouts($data, $object, $customLayoutToSelect = null, $layoutsToRemove = [])
    {
        if ($customLayoutToSelect !== null) {
            $data['currentLayoutId'] = $customLayoutToSelect;
            $customLayout = CustomLayout::getById($customLayoutToSelect);
            if (null !== $customLayout) {
                $data['layout'] = $customLayout->getLayoutDefinitions();
                Service::enrichLayoutDefinition($data["layout"], $object);
            }
        }

        if (!empty($layoutsToRemove)) {
            $validLayouts = $data["validLayouts"];
            foreach ($validLayouts as $key => $validLayout) {
                if (in_array($validLayout['id'], $layoutsToRemove)) {
                    unset($validLayouts[$key]);
                }
            }
            $data["validLayouts"] = array_values($validLayouts);
        }

        return $data;
    }
}
