<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\OrderInventory\Notifier;

use Lifestyle\Pimcore\OrderInventory\Configuration\NotifierConfiguration;
use Pimcore\Model\DataObject;
use Psr\Log\LoggerInterface;
use Symfony\Component\Templating\EngineInterface;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Class OnHandLimitNotifier
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\OrderInventory\Notifier
 */
class OnHandLimitNotifier
{
    /**
     * @var NotifierConfiguration
     */
    private $configuration;

    /**
     * @var \Swift_Mailer
     */
    private $mailer;

    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * @var EngineInterface
     */
    private $templateEngine;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * OnHandLimitNotifier constructor.
     * @param NotifierConfiguration $configuration
     * @param \Swift_Mailer $mailer
     * @param TranslatorInterface $translator
     * @param EngineInterface $templateEngine
     * @param LoggerInterface $logger
     */
    public function __construct(
        NotifierConfiguration $configuration,
        \Swift_Mailer $mailer,
        TranslatorInterface $translator,
        EngineInterface $templateEngine,
        LoggerInterface $logger
    ) {
        $this->configuration = $configuration;
        $this->mailer = $mailer;
        $this->translator = $translator;
        $this->templateEngine = $templateEngine;
        $this->logger = $logger;
    }

    /**
     * @param DataObject\AbstractObject $product
     * @param int $previousOnHand
     * @param int $currentOnHand
     */
    public function onSell(DataObject\AbstractObject $product, int $previousOnHand, int $currentOnHand)
    {
        if (
            $this->configuration->isEnabled() &&
            $this->configuration->getRecipients() &&
            $previousOnHand > $this->configuration->getMinOnHold() &&
            $currentOnHand <= $this->configuration->getMinOnHold()
        ) {
            $messageBody = $this->templateEngine->render(
                '@LifestylePimcoreOrderInventory/notifier/on_hand_limit_notifier.html.twig',
                ['product' => $product]
            );

            (new \Pimcore\Mail())
                ->setFrom([$this->configuration->getSenderEmail() => $this->configuration->getSenderName()])
                ->setTo($this->configuration->getRecipients())
                ->setSubject($this->translator->trans('product_on_hand_notifier_subject'))
                ->setBody($messageBody, 'text/html')
                ->send();

            $this->logger->debug(sprintf(
                'Minimum OnHand Notification email for product %d should has been sent to %d recipients.',
                $product->getId(),
                count($this->configuration->getRecipients())
            ));
        }
    }
}
