<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\OrderInventory\Handler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\OrderInventory\Exception\ExceptionInterface;
use Lifestyle\Pimcore\OrderInventory\Exception\InvalidArgumentException;
use Lifestyle\Pimcore\OrderInventory\Exception\NotFoundException;
use Lifestyle\Pimcore\OrderInventory\Model\ProductMapper;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Pimcore\Model\DataObject;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\ConstraintViolationInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Component\Validator\ConstraintViolationListInterface;

/**
 * Class AbstractHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\OrderInventory\Handler
 */
abstract class AbstractHandler implements HandlerInterface
{
    /**
     * @var ProductMapper
     */
    private $mapper;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * AbstractHandler constructor.
     * @param ProductMapper $mapper
     * @param ValidatorInterface $validator
     */
    public function __construct(ProductMapper $mapper, ValidatorInterface $validator)
    {
        $this->mapper = $mapper;
        $this->validator = $validator;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws ExceptionInterface
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        $productData = $this->mapper->mapProduct($dataCollector);
        $violations = $this->validator->validate($productData);
        if (0 < count($violations)) {
            throw new InvalidArgumentException(
                $this->violationsToString($violations),
                Response::HTTP_BAD_REQUEST
            );
        }

        $product = DataObject::getById($productData->getId(), true);
        if (null === $product) {
            throw new NotFoundException(
                sprintf('Object (%d) not found.', $productData->getId()),
                Response::HTTP_NOT_FOUND
            );
        }

        return $this->execute($product, $productData->getChangedQuantity(), $productData->getOnHold());
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return
            $dataCollector->hasItem('event') &&
            $dataCollector->getItemValue('event') === $this->getResponsibleEvent();
    }

    /**
     * @return string
     */
    protected abstract function getResponsibleEvent(): string;

    /**
     * @param ConstraintViolationListInterface $violations
     * @return string
     */
    private function violationsToString(ConstraintViolationListInterface $violations): string
    {
        return implode(
            ' ',
            array_map(
                function (ConstraintViolationInterface $violation) {
                    return $violation->getMessage();
                },
                iterator_to_array($violations)
            )
        );
    }

    /**
     * @param DataObject\AbstractObject $product
     * @param int $changedQuantity
     * @param int|null $onHold
     * @return bool
     */
    protected abstract function execute(DataObject\AbstractObject $product, int $changedQuantity, ?int $onHold): bool;
}
